REBOL [
    Title:   "A Rebol Text Reader" 
    Name:    'more
    Author:  "Dick Whiting"
    Email:   dwhiting@europa.com
    Home:    http://www.europa.com/~dwhiting
    Date:    22-Jan-1999
    File:    %more.r
    Version: 1.0.4

    Changes: {
        Return now does a down one page.
        Added commands #,E,C -- see cmdhelp.
        Added ability to process strings without newline breaks.
        Find and Save can now accept their arguments without prompting.
        Auto-converts CR/LF to your platform's newline characters.
        Save and print process the formatted output.
        Tidied up some of the code.
    }

    Needs: {
        Uses functions from dwstrings.rlib version 1.1.3, available at:
        http://www.europa.com/~dwhiting/satyre/progrebol.html
    }

    Purpose: {
        A "more" style text reader.
    }

    Comments: {
        Quite a few:) Jeff said he wanted a script "that would, say, show
        my cousin Timmy who's a BASIC programmer how to do some clever
        things in REBOL", so I documented it a bit more than I would for
        folks on the Rebol list. Hi there Timmy;)
    }

    Setup: {
        The variables: pagelen, pagewidth, clrscr, and numbered 
        may be changed within the code to set your personal defaults.
        Search for 'SETUP' and read comments about defaults.
    }
]

more: func [strfile /pl pagelen /pw pagewidth] [
    {
;**************************************************************************;
;*                    A simple MORE style text viewer.                    *;
;**************************************************************************;
;*                                                                        *;
;*  Format:                                                               *;
;*                                                                        *;
;*   more strfile /pl pagelen /pw pagewidth                               *;
;*                                                                        *;
;*  Example usages:                                                       *;
;*                                                                        *;
;*   more a-file-read-into-a-string!                                      *;
;*   more read %test.doc                                                  *;
;*   more read http://www.rebol.com/library/urlencode.r                   *;
;*                                                                        *;
;*   more/pl a-file 24  will use 24 lines as the page length.             *;
;*                                                                        *;
;*   more/pw a-file 40  breaks a file without newlines into 40 char lines *;
;*                                                                        *;
;*   Can be combined with other functions that return a string:           *;
;*                                                                        *;
;*   more libinfo dwstrings comment  (view the comment in dwstrings.rlib) *;
;*                                                                        *;
;*                                                                        *;
;**************************************************************************;
    }

;**************************************************************************;
;*                 Hide these words from the outside world                *;
;**************************************************************************;

    use [ctop cbot str strblk oldpos newpos blkend numbered 
         ans  arg cmd  commands cmdhelp clrscr srch linestr crlf] [


;**************************************************************************;
;*                             SETUP defaults                             *;
;*                                                                        *;
;*  The following variables can be changed to your choice of defaults:    *;
;*                                                                        *;
;*  pagelen   -- Set the number of lines (-1) for a page                  *;
;*  pagewidth -- Set the column width for files/strings without newlines  *;
;*  numbered  -- Set numbered: true for line numbering as the default     *;
;*  clrscr    -- On the Amiga, print '0C'x acts as a clear screen.        *;
;*               To enable it, delete or comment out 'clrscr: none!'      *;
;*               and uncomment 'clrscr: char! 12'                         *;
;*                                                                        *;
;**************************************************************************;

        if not pl [pagelen: 39]         ; default page length
        else [pagelen: pagelen - 1]

        if not pw [pagewidth: 80]       ; default page width

        numbered: false                 ; default un-numbered display

;       clrscr: none!          ; default is no screen clearing
        clrscr: make char! 12  ; uncomment this to use FF to clear screen
                               ; between printing pages of lines..

;**************************************************************************;
;*                       Basic setup of some values                       *;
;**************************************************************************;

        ctop: 1
        cbot: pagelen

        str:    make string! strfile
        strblk: make block! 0
        oldpos: make integer! 0
        newpos: make integer! 0
        cmd: "start"
        srch: make string! 0

        cmdhelp: {
;*************************************************************************;
;*                                                                       *;
;*  Commands:                                                            *;
;*                                                                       *;
;*   #         toggle line NUMBERING on and off                          *;
;*   B         pages to BOTTOM of file                                   *;
;*   C         toggle CLEAR SCREEN mode .. may only work on Amiga        *;
;*   D         pages DOWN one page length                                *;
;*   D 20      pages DOWN 20 lines                                       *;
;*   E         EXECUTE a REBOL command                                   *;
;*   F         FIND a line containing string -- searches from top        *;
;*   H         HELP on available commands                                *;
;*   L         displays the current page LENGTH                          *;
;*   L 45      set page LENGTH to 45 lines                               *;
;*   N         find NEXT occurrence of search string from current line   *;
;*   P         PRINT file to %prt -- functions ONLY on Amiga version     *;
;*   Q         QUIT more                                                 *;
;*   T         TOP page of file                                          *;
;*                                                                       *;
;*   S         SAVE a file to disk -- prompts for filename               *;
;*             (to cancel press return without entering anything)        *;
;*             example of filenames:                                     *;
;*                  T:/test.doc  -- save to volume T: as test.doc        *;
;*                  test.doc     -- save in rebol's home directory       *;
;*                                                                       *;
;*   U         pages UP one page length (40 line default)                *;
;*   U 10      pages UP 10 lines                                         *;
;*                                                                       *;
;*************************************************************************;
        }

;**************************************************************************;
;*                   Recognized commands and their code                   *;
;**************************************************************************;
;* There are quite a few ways of doing this type of thing in REBOL,       *;
;* but this one seemed appropriate here. The main code tries to locate    *;
;* the command value ("U") and if found will then execute the block of    *;
;* code following the match. Makes it simple to add/modify a command.     *;
;* Besides, have been looking for a place to use the select function:)    *;
;**************************************************************************;

        commands: [
        
            "U" [ctop: maximum 1 (ctop - arg)
                 cbot: minimum (ctop + pagelen) blkend
                ]
            
            "D" [ctop: ctop + arg
                 cbot: minimum (ctop + pagelen) blkend
                ]
            
            "H" [print cmdhelp]
            
            "T" [ctop: 1
                 cbot: minimum (ctop + pagelen) blkend
                ]
            
            "B" [ctop: maximum 1 (blkend - pagelen)
                 cbot: blkend
                ]
            
            "L" [if arg = pagelen [print pagelen] else [pagelen: arg]
                 cmd: "H"
                ]
            
            
            "P" [
                    if (fourth rebol/version) = 1 [
                        write %prt:/ form [str make char! 12]
                    ] else [print ["Can't print from Rebol version" rebol/version]]
                    cmd: "H"
                ]
            
            
            "S" [
                    if none? arg [
                        prin "Enter file name for save: " 
                        savename: make string! input
                    ] else [savename: trim/auto arg]

                    if savename <> "" [
                        if exists? make file! savename [
                            print [savename "already exists.. Overwrite it?"]
                            prin "(Y/n): "
                            ans: make string! input
                            if ans = "N" [print "Save canceled"]
                            else [
                               write make file! savename str
                               print ["File saved as:" savename]
                            ]
                        ] else [
                            write make file! savename str
                            print ["File saved as:" savename]
                        ]
                    ]
                    else [print "File not saved"]
                    cmd: "H"
                ]
            
            "F" [
                    if none? arg [
                        prin "Enter search string: "
                        srch: make string! input
                        print ""
                    ] else [srch: trim/auto arg]
                    sfound: false
                    if srch <> "" [
                        for n 1 blkend 1 [
                            if found? find pick strblk n srch [
                                ctop: n
                                cbot: minimum (ctop + pagelen) blkend
                                sfound: true
                                break
                            ]
                        ]
                    ]
            
                    if not sfound [
                        if srch <> "" [print ["Couldn't find:" srch]]
                        srch: ""
                        cmd: "H"
                    ]
                ]
            
            "N" [ 
                    if srch = "" [do select commands "F"]
                    else [
                        sfound: false
                        for n (ctop + 1) blkend 1 [
                            if found? find pick strblk n srch [
                                ctop: n
                                cbot: minimum (ctop + pagelen) blkend
                                sfound: true
                                break
                            ]
                        ]
                        if not sfound [
                            print ["Couldn't find:" srch]
                            cmd: "H"
                        ]
                    ]
                ]      

            "E" [do arg
                 cmd: "H"]

            "#" [numbered: not numbered]

            "C" [if none? clrscr [clrscr: make char! 12] else [clrscr: none!]]

        ]
      
        print "Formatting lines..."

;**************************************************************************;
;*         Convert CR/LF to newlines, if not a Wxx Rebol version          *;
;**************************************************************************;

        crlf: form [make char! 13 make char! 10]
        crlf: head remove/part skip crlf 1 1

        if found? find str crlf [
            if (fourth rebol/version) <> 3 [
                str: replstr str crlf form newline
            ]
        ]            

;**************************************************************************;
;*       Chop the string into individual lines and store in a block       *;
;*   If the string has no newline characters, then chop it into pw chars  *;
;*      broken at spaces, or as a last resort just at pw characters.      *;
;**************************************************************************;

        if none? strblk: scanstr str newline [
            strblk: scanstr wraptxt str pagewidth newline
        ]

        blkend: length? strblk
        if cbot > blkend [cbot: blkend]

;**************************************************************************;
;*              Display page of lines and wait for user input             *;
;**************************************************************************;
;*                                                                        *;
;*  This is the actual code. It just loops on an input statement waiting  *;
;*  for you to enter "Q" to quit. The actual commands executed are in the *;
;*  commands block and are pretty obvious. Mostly they just adjust the top*;
;*  and bottom line to view and try to ensure valid values.               *;
;*  REBOL currently has no real error trapping, so an invalid filename for*;
;*  the S(ave) command will break the script.                             *;
;**************************************************************************;

        while [cmd <> "Q"] [

            if cmd <> "H" [
                if not none? clrscr [prin clrscr]
                for linenum ctop cbot 1 [
                    if numbered [
                        linestr: form linenum
                        linestr: insert/dup linestr "0" (6 - (length? linestr))
                        prin [head linestr " "]
                    ]
                    print [pick strblk linenum]
                ]
            ]

            print ""
            prin "==> "
            ans: form trim/auto make string! input
            print ""

            if ans = "" [cmd: "D"]
            else [cmd: form first ans]

            arg: find ans " "
            if found? find "UDL" cmd [
                if not none? arg [arg: make integer! trim/auto arg] 
                if none? arg [arg: pagelen]
            ]

            if found? find commands cmd [do select commands cmd]
            else [
                if cmd <> "Q" [
                    print ""
                    print {Unknown command -- Enter "H" for HELP}
                    print ""
                    cmd: "H"
                ]
            ]
        ]
        return ""  ; used to hide FALSE on Quitting
    ]              ; end of USE
]                  ; end of MORE

""
