REBOL [
   Title:   "Install dwdate.rlib pieces"
   Author:  "Dick Whiting"
   Email:   dwhiting@europa.com
   Home:    http://www.europa.com/~dwhiting
   Date:    30-Dec-1998
   File:    %install-dwdate.r
   Version: 1.0.0

   Usage: {
   
      Within the same directory that REBOL exists:

         Create a directory named "library/datetime/"
         Create a directory named "help/datetime/dwdate/"

      Copy this file into your main REBOL directory

      Start REBOL interactively and enter:  

         do %install-dwdate.r

      This script will do the following:

      1) Save file %library/datetime/dwdate.locale           (sample locale)
      2) Save file %library/datetime/dwdate.rlib             (the functions)
      3) Save help files %help/datetime/dwdate/*.help        (help files)
      4) Save example files %help/datetime/dwdate/*.example  (example files)

      You can discard this script after the installation has been done.      
      
   }

   Purpose: {

      Handles installing dwdate.rlib and its corequisites

   }

   Comment: {

      The "working part of the install script can be located by
      searching for the string "INSTALLER STARTS HERE" -- at bottom.

   }

]

;**************************************************************************;
;*                     Variables used during install                      *;
;**************************************************************************;
script-help: "datetime/dwdate/"

;**************************************************************************;
;*                          Introduction message                          *;
;**************************************************************************;
;*                       Set Intromsg: "" to disable                      *;
;**************************************************************************;

Intromsg: {

This script installs all of the pieces for:

   Title:   "A collection of (A)Rexx-like string functions"
   Author:  "Dick Whiting"
   Email:   dwhiting@europa.com
   Home:    http://www.europa.com/~dwhiting
   Date:    30-Dec-1998
   File:    %library/datetime/dwdate.rlib
   Version: 1.0.0


This package is meant to replace rexxsfuncs.r. If you have that version, 
then you may delete it after installing this one.

This script will do the following:

   1) Save file %library/datetime/dwdate.locale           (sample locale)
   2) Save file %library/datetime/dwdate.rlib             (the functions)
   3) Save help files %help/datetime/dwdate/*.help        (help files)
   4) Save example files %help/datetime/dwdate/*.example  (example files)

Press "Q" to quit or any other key to continue
}

;**************************************************************************;
;*                       Changes since last version                       *;
;**************************************************************************;
;*                     Set Changesmsg: "" to disable                      *;
;**************************************************************************;

Changesmsg: ""

;**************************************************************************;
;*             User has to do these by hand before installing             *;
;**************************************************************************;
;*                      Set ByHandMsg: "" to disable                      *;
;**************************************************************************;

ByHandMsg: {

Before continuing with this install script, you must create the following
directories within your main REBOL directory:

   library/datetime/
   help/datetime/dwdate/

Press "Q" to quit or any other key to continue
}

;**************************************************************************;
;*                     Final message at end of install                    *;
;**************************************************************************;
;*                       Set FinalMsg: "" to disable                      *;
;**************************************************************************;

FinalMsg: {

If you have not installed the dwsupport.rlib package, then you should do that
now. It provides the functions supporting external help and example files, 
and the tracking of loaded libraries. When you have done that--

Add the following line to your user.r script, if you want these functions
available each time you start REBOL:

   loadlib "datetime/dwdate"


These functions also require functions from the dwstrings.rlib package.

To enable language LOCALIZATION for this rlib, refer to dwdate.locale and
follow the instructions within it.

}

;**************************************************************************;
;*                      List of required directories                      *;
;*             List is pairs with location and subdir needed              *;
;**************************************************************************;
;*                    Set required-dirs: [] to disable                    *;
;**************************************************************************;

required-dirs: [

   "library" "datetime" 
   "help"    "datetime/dwdate"
]

;**************************************************************************;
;*                          Files to be installed                         *;
;* List is triad with path type, file path, and file included as a string *;
;**************************************************************************;

files-to-install: [ 

   "library" 
   "datetime/dwdate.rlib" 
   {REBOL [
   Title:   "Date manipulation functions"
   Author:  "Dick Whiting"
   Email:   dwhiting@europa.com
   Home:    http://www.europa.com/~dwhiting
   Date:    30-Dec-1998
   File:    %library/datetime/date.rlib
   Version: 1.0.0

   Requirements: {

      Uses functions from dwstrings.rlib and dwsupport.rlib
      
   }

   Purpose: {

      Provide date manipulation functions

   }

   Comment: {
      
      Dates are calculated as if the Gregorian calendar exists back to 
      1-Jan-0000. Values before then will yield unpredictable results.

      Most of these functions have an /OS refinement or version, that allows
      for Julian or "old-style" dates as input or output.

      Dates and therefore date manipulation turns out to be VERY complicated.
      These functions will work as expected for dates after the change from
      the Julian to the Gregorian calendar. Prior to that date, you MUST
      know what you want and what calendar you are dealing with. 

      So...don't try to figure out what day of the week a date was if you
      don't know WHEN they went to the Gregorian or if the date predates
      Oct 15, 1582, which on THAT date "yesterday" was Oct 4, 1582. The 
      British Empire (including the US) didn't change until 1752 and China
      was the last major power to do so, converting in 1949. 

      Oh...and the New Year used to be March 25th to really confuse things.
   }

   Functions: {

      CALENDAR       print a calendar for a month and year (Gregorian or Julian)
      DATESTR        returns a date in a variety of formats as a string
      GETDAYS        return the nth weekday (e.g. 3rd Thursday) for month/year
      GREG2JUL       convert a Gregorian date to a Julian date
      INCRDATE       increment or decrement date or datetime values

      IS-JULIAN-DATE test if a string describes a valid Julian date
      JULIAN-DATE?   alias for is-julian-date               

      IS-LEAP-YEAR   test if year is a valid Gregorian leap year
      LEAP-YEAR?     alias for is-leap-year

      JUL2GREG       convert a Julian date to a Gregorian date
      LASTDAY        get last day for month and year (Gregorian or Julian)
      OSWEEKDAY      get what day of the week for a Julian ("old-style") date
      WEEKDAY        get what day of the week for a Gregorian date
      
   }

   Variables: {

      Creates these global variables:
      
      names-of-days     -- full names for days of the week
      abbrev-of-days    -- 3 character abbreviations for days
      names-of-months   -- full names for months of the year
      abbrev-of-months  -- 3 character abbreviations for months
      lengths-of-months -- number of days in each non-leap year month
      base-sunday       -- date of a Sunday in the Gregorian calendar
      OSbase-sunday     -- date of a Sunday in the Julian calendar

   }

   Bugs: {
      
      As of REBOL 1.0.3.x adding and subtracting days to a date wraps
      at approximately 32,000 days. This bug is not corrected in the
      INCRDATE function, as I expect it to be fixed within REBOL in the
      near future. Also, time increments above 99,999 result in errors.

   }

   Uninstall: [
   
      "library" "datetime/dwdate.*"
      "help"    "datetime/dwdate"
      "user"    "references to dwdate"

   ]
]

;**************************************************************************;
;*  For localization refer to dwdate.locale file                          *;
;**************************************************************************;

names-of-days: [
      "Sunday" 
      "Monday" 
      "Tuesday" 
      "Wednesday" 
      "Thursday" 
      "Friday" 
      "Saturday"
]
 
abbrev-of-days: [
      "Sun" 
      "Mon" 
      "Tue" 
      "Wed" 
      "Thu" 
      "Fri" 
      "Sat"
]
 
names-of-months: [
      "January" 
      "February" 
      "March" 
      "April" 
      "May" 
      "June" 
      "July" 
      "August" 
      "September" 
      "October" 
      "November" 
      "December"
]

abbrev-of-months: [
      "Jan" 
      "Feb" 
      "Mar" 
      "Apr" 
      "May" 
      "Jun" 
      "Jul" 
      "Aug" 
      "Sep" 
      "Oct" 
      "Nov" 
      "Dec"
]

;**************************************************************************;
;*                       Some basic data                                  *;
;**************************************************************************;

lengths-of-months: [31 28 31 30 31 30 31 31 30 31 30 31]

base-sunday: 20-Dec-1998   ;; a recent Sunday used for weekday function
OSbase-sunday: 04-Jan-0000 ;; base Sunday for Julian calendars      

;**************************************************************************;
;*              Test if year is a valid Gregorian leap year               *;
;**************************************************************************;

is-leap-year: func [year] [

   if (year // 400) = 0 [return true]
   if (year // 100) = 0 [return false]
   if (year // 4)   = 0 [return true]
   return false
]

alias 'is-leap-year "leap-year?"

;**************************************************************************;
;*     Check Julian string! date in "dd/mm/yyyy" format for validity      *;
;**************************************************************************;

is-julian-date: func [datestr] [

   use [jdate jday jmon jyear] [

      if (none? jdate: scanstr datestr "/") [return false]      
      if ((length? jdate) <> 3) [return false] 

      jday:  make integer! first  jdate
      jmon:  make integer! second jdate
      jyear: make integer! third  jdate

      if any [(none? jday) (none? jmon) (none? jyear)] [return false]

      if all [(jmon > 0) (jmon < 13) (jday > 0) (jyear >= 0)] [
         if all [(jmon = 2) ((jyear // 4) = 0) (jday < 30)] [return true]
         if (jday <= (pick lengths-of-months jmon)) [return true]
      ]
      return false
   ]
]

alias 'is-julian-date "julian-date?"

;**************************************************************************;
;*                  return formatted date STRING                          *;
;**************************************************************************;
;*                                                                        *;
;* Formats:                                                               *;
;*                                                                        *;
;* euro      -- ddmmyyyy i.e. 25/01/1999 for 25-Jan-1999                  *;
;* euro6     -- ddmmyy   i.e.   25/01/99 for 25-Jan-1999                  *;
;* julian    -- yyyyddd  i.e.    1999025 for 25-Jan-1999                  *;
;* julian5   -- yyddd    i.e.      99025 for 25-Jan-1999                  *;
;* sorted    -- yyyymmdd i.e.   19990125 for 25-Jan-1999                  *;
;* usa       -- mmddyyyy i.e. 01/25/1999 for 25-Jan-1999                  *;
;* usa6      -- mmddyy   i.e.   01/25/99 for 25-Jan-1999                  *;
;*                                                                        *;
;* fmtstr    -- see help (or code) for replacement patterns               *;
;*              note: /sep " " won't work -- translate returned string    *;
;**************************************************************************;

datestr: func [datein /euro /euro6 /julian /julian5 /sorted 
                      /usa /usa6 /sep schar /fmt fmtstr ] [

   if not date? datein [return none!]

   use [syear edays timepart stime] [

      if not sep [schar: "/"]

      if sorted [
         return compress form [
            third datein
            right/pad form second datein 2 "0" 
            right/pad form first datein 2 "0"
         ]  
      ]
      if euro [
         return compress form [
            right/pad form first datein 2 "0" 
            schar 
            right/pad form second datein 2 "0" 
            schar 
            form third datein
         ]
      ]
      if julian [
         syear: make date! compress form ["1/1/" third datein]
         edays:  right/pad form [datein - syear + 1] 3 "0"
         return compress form [
            third datein
            edays
         ]
      ]
      if usa [
         return compress form [
            right/pad form second datein 2 "0" 
            schar 
            right/pad form first datein 2 "0" 
            schar 
            form third datein
         ]  
      ]
      if euro6 [
         return compress form [
            right/pad form first datein 2 "0" 
            schar 
            right/pad form second datein 2 "0" 
            schar 
            right/pad form third datein 2 "0"
         ]  
      ]
      if julian5 [
         syear: make date! compress form ["1/1/" third datein]
         edays:  right/pad form [datein - syear + 1] 3 "0"
         return compress form [
            right/pad form third datein 2 "0"
            edays
         ]
      ]
      if usa6 [
         return compress form [
            right/pad form second datein 2 "0" 
            schar 
            right/pad form first datein 2 "0" 
            schar 
            right/pad form third datein 2 "0"
         ]  
      ]


;**************************************************************************;
;*                   Replace %keys with datetime values                   *;
;*          %xxxx0 says to right justify and zero fill to length          *;
;*          %xxxx  says to form without leading zeroes                    *;
;**************************************************************************;

      if fmt [
         fmtstr: replstr/case fmtstr "%yyyy0" right/pad form third datein 4 "0"
         fmtstr: replstr/case fmtstr "%yyyy"  form third datein
         fmtstr: replstr/case fmtstr "%yy0"   right/pad form third datein 2 "0" 
         fmtstr: replstr/case fmtstr "%yy"    form third datein
         fmtstr: replstr/case fmtstr "%MMM"   pick names-of-months second datein
         fmtstr: replstr/case fmtstr "%mmm"   pick abbrev-of-months second datein
         fmtstr: replstr/case fmtstr "%mm0"   right/pad form second datein 2 "0" 
         fmtstr: replstr/case fmtstr "%mm"    form second datein

         fmtstr: replstr/case fmtstr "%DDD" weekday datein
         fmtstr: replstr/case fmtstr "%ddd" pick abbrev-of-days weekday/n datein
         fmtstr: replstr/case fmtstr "%dd0" right/pad form first datein 2 "0" 
         fmtstr: replstr/case fmtstr "%dd"  form first datein

         if (pos "%doy0" fmtstr) > 0 [
            syear: make date! compress form ["1/1/" third datein]
            edays:  right/pad form [datein - syear + 1] 3 "0"
            fmtstr: replstr/case fmtstr "%doy0" edays
         ]
         if (pos "%doy" fmtstr) > 0 [
            syear: make date! compress form ["1/1/" third datein]
            edays:  form [datein - syear + 1]
            fmtstr: replstr/case fmtstr "%doy" edays
         ]

;**************************************************************************;
;*                      Handle time portion of datein                     *;
;*  Removes partial seconds from value -- Defaults to 0:00:00 if no time  *;
;**************************************************************************;

         if (pos "%" fmtstr) > 0 [

            if (fourth datein) < 0:00 [
               timepart: make time! 0:00:00
               stime: make string! "00:00:00"
            ]
            else [
               timepart: make time! fourth datein
               timepart: timepart - (timepart // 1)
               stime: form timepart
               if (length? stime) < 5 [
                  stime: head insert tail stime ":00"
               ]
               stime: right/pad stime 8 "0"
            ]

            if (pos "%time8" fmtstr) > 0 [
               fmtstr: replstr/case fmtstr "%time8" stime
            ]
   
            if (pos "%time5" fmtstr) > 0 [
               fmtstr: replstr/case fmtstr "%time5" substr stime 1 5
            ]

            if (pos "%time" fmtstr) > 0 [
               fmtstr: replstr/case fmtstr "%time" form timepart
            ]
   
            if (pos "%hr0" fmtstr) > 0 [
               fmtstr: replstr/case fmtstr "%hr0" 
                  right/pad form first timepart 2 "0"
            ]
            if (pos "%hr" fmtstr) > 0 [
               fmtstr: replstr/case fmtstr "%hr" form first timepart
            ]
            if (pos "%min0" fmtstr) > 0 [
               fmtstr: replstr/case fmtstr "%min0" 
                  right/pad form second timepart 2 "0"
            ]
            if (pos "%min" fmtstr) > 0 [
               fmtstr: replstr/case fmtstr "%min" form second timepart
            ]
            if (pos "%sec0" fmtstr) > 0 [
               fmtstr: replstr/case fmtstr "%sec0" 
                  right/pad form third timepart 2 "0"
            ]
            if (pos "%sec" fmtstr) > 0 [
               fmtstr: replstr/case fmtstr "%sec" form third timepart
            ]
         ]
         
         return fmtstr
      ]
   ]
]


;**************************************************************************;
;*                   Get what day of the week for a date                  *;
;**************************************************************************;

weekday: func [datein /n] [

   use [nday] [

      if datein > base-sunday [nday: 1 + (datein - base-sunday // 7)]
      else [
         nday: 8 - abs (datein - base-sunday // 7)
         if nday = 8 [nday: 1]
      ]
      if n [return nday]
      return pick names-of-days nday
   ]
]


;**************************************************************************;
;*          Weekday for Julian calendar (also called "Old Style")         *;
;**************************************************************************;
;*   Extra leap year days need to be passed as a string e.g. "29/2/1500"  *;
;**************************************************************************;

OSweekday: func [datein /n] [

   use [jdate jday jmon jyear cent adj nday leap] [

      leap: false
      if string? datein [
         if is-julian-date datein [
            jdate: scanstr datein "/"
            jday:  make integer! first  jdate
            jmon:  make integer! second jdate
            jyear: make integer! third  jdate
            if all [(jmon = 2) (jday = 29) 
                    ((jyear // 400) <> 0) ((jyear // 100) = 0)] [
                  leap: true
                  jday: 28
            ]
            datein: make date! compress form
                        [jday "/" jmon "/" right/pad form jyear 4 "0"]
                  
         ]
         else [return none!]
      ]
      else [
         if not date? datein [return none!]
      ]

      if datein > OSbase-sunday [nday: (datein - OSbase-sunday // 7)]
      else [
         nday: 7 - abs (datein - OSbase-sunday // 7)
      ]
      if leap [nday: nday + 1]

      cent: make integer! third datein

      if all [(cent > 0)  ((cent // 100) = 0)  ((cent // 400) <> 0) 
         (((second datein) * 100 + (first datein)) < 229)] [cent: cent - 100]

      cent: make integer! cent / 100 
      adj: 0 
      for num 1 cent 1 [
         if not ((num // 4) = 0) [adj: adj + 1]
      ]

      nday: 1 + ((nday + adj) // 7)

      if n [return nday]
      return pick names-of-days nday
   ]
]

;**************************************************************************;
;*                     Get last day for month and year                    *;
;*                    /OS for "old-style" Julian dates                    *;
;**************************************************************************;

lastday: func [month year /OS] [

   if not OS [
      if (is-leap-year year) and (month = 2) [return 29]
      return pick lengths-of-months month
   ]
   else [
      if ((year // 4) = 0) and (month =  2) [return 29]
      return pick lengths-of-months month
   ]
]

;**************************************************************************;
;*            increment/decrement a date/time by an amount                *;
;**************************************************************************;
;*  default action is to roll-over extra days to new month                *;
;*  /keep to retain month -- use last day of month                        *;
;*  /none to return none! when a roll-over would occur                    *;
;**************************************************************************;

incrdate: func [datein /d dincr /w wincr /m mincr /y yincr 
                       /hr hincr /min mnincr /sec sincr /none /keep /roll] [

   use [t+ tyear tmon tday ttime stime] [

      if not date? datein [return none!]

      t+: func [incr] [
         if incr >= 0 [:add]
         else [:subtract]
      ]
 
      if y or m [
         tday: first datein
         tmon: second datein
         tyear: third datein
         ttime: fourth datein
         
         if ttime < 0:00:00 [stime: ""] else [stime: form ["/" ttime]]

         if y [tyear: tyear + yincr]

         if m [
            tmon: tmon + mincr
            if tmon > 12 [
               tyear: tyear + (make integer! (tmon / 12))
               tmon: tmon // 12
            ]
         ] 

         datein: make date! compress form 
                    [tday "/" tmon "/" right/pad form tyear 4 "0" stime]

         if none? datein [
            if none [return none!]
            if keep [
               tday: lastday tmon tyear
            ]
            datein: make date! reduce [tday tmon tyear ttime]
         ]
      ]

      if w [datein: do t+ wincr datein (7 * wincr)]
      if d [datein: do t+ dincr datein abs dincr]
      if ttime >= 0:00:00 [
         if hr  [datein: do t+ hincr datein make time! compress form [abs hincr ":00:00"]]
         if min [datein: do t+ mincr datein make time! compress form ["0:" abs mnincr ":00"]]
         if sec [datein: do t+ sincr datein make time! compress form ["00:00:" abs sincr]]
      ]

      return datein
   ]
]

;**************************************************************************;
;*                   Convert a Gregorian date to Julian                   *;
;*                   Undoes the century leap corrections                  *;
;**************************************************************************;
;*     Extra leap year days are returned as a string e.g. "29/2/1500"     *;
;**************************************************************************;

greg2jul: func [datein] [

   use [gday gmon gyear cent adj eom] [

      if date? datein [
         gday:  first  datein
         gmon:  second datein
         gyear: third  datein
      ]
      else [return none!]

      adj: 0
      if  (gyear < 100) or
         ((gyear = 100) and (gmon < 3)) [gday: gday + 2]
      else [
         if  (gyear < 200) or
            ((gyear = 200) and (gmon < 3)) [gday: gday + 1]
         else [
            cent: make integer! gyear / 100
            adj: 0
            for n 3 cent 1 [if not ((n // 4) = 0) [adj: adj + 1]]

            if all [((gyear // 100)  = 0)
                 ((gyear // 400) <> 0) (gmon < 3)] [adj: adj - 1]
         ]
      ]

      gday: gday - adj

;**************************************************************************;
;*                  Handle adjustments to gregorian date                  *;
;**************************************************************************;

      eom: lastday/OS gmon gyear
      while [gday > eom] [
         gday: gday - eom
         gmon: gmon + 1
         if gmon > 12 [
            gmon: 1
            gyear: gyear + 1
         ]
         eom: lastday/OS gmon gyear
      ]

      while [gday < 1] [
         gmon: gmon - 1
         if gmon < 1 [
            gmon: 12
            gyear: gyear - 1
         ]
         eom: lastday/OS gmon gyear
         gday: eom + gday
      ]

      if all [(gmon = 2) (gday = 29) (not is-leap-year gyear)] [
         return compress form [gday "/" gmon "/" right/pad form gyear 4 "0"]
      ]
      else [
         return make date! reduce [gday gmon gyear]
      ]
   ]
]

;**************************************************************************;
;*               Convert a Julian date to a Gregorian date                *;
;**************************************************************************;
;*   Extra leap year days need to be passed as a string e.g. "29/2/1500"  *;
;**************************************************************************;

jul2greg: func [datein] [

   use [temp jdate jday jmon jyear cent adj eom] [

      if string? datein [
         if is-julian-date datein [
            jdate: scanstr datein "/"
            jday:  make integer! first  jdate
            jmon:  make integer! second jdate
            jyear: make integer! third  jdate
         ]
         else [return none!]
      ]
      else [
         if date? datein [
            jday:  first  datein
            jmon:  second datein
            jyear: third  datein
         ]
         else [return none!]
      ]

      adj: 0
      if  (jyear < 100) or
         ((jyear = 100) and (jmon < 3)) [adj: - 2]
      else [
         if  (jyear < 200) or
            ((jyear = 200) and (jmon < 3)) [adj: - 1]
         else [
            cent: make integer! jyear / 100
            adj: 0
            for n 3 cent 1 [if not ((n // 4) = 0) [adj: adj + 1]]
            if all [((jyear // 100)  = 0)
                    ((jyear // 400) <> 0)
                     (jmon < 3)] [adj: adj - 1]
         ]
      ]

      return make date! reduce [(jday + adj) jmon jyear]

   ]
]

;**************************************************************************;
;*             Return the nth weekday (3rd Thursday) of month             *;
;**************************************************************************;

getdays: func [dayofwk month year /f /l /n num /full] [

   use [dayone firstday eom dayblk] [

      if string? dayofwk [
         dayofwk: find names-of-days dayofwk
         if none? dayofwk [return none!]
         dayofwk: index? dayofwk
      ]

      dayone: weekday/n make date! reduce [1 month year]
      if dayone = dayofwk [firstday: 1]
      else [
         if dayone < dayofwk [firstday: 1 + (dayofwk - dayone)]
         else [firstday: 8 - (dayone - dayofwk)]
      ]

      eom: lastday month year

      dayblk: make block! 0
      for nday firstday eom 7 [
         if full [insert tail dayblk make date! reduce [nday month year]]
         else [insert tail dayblk nday]
      ]

      dayblk: head dayblk

      if f [return first dayblk]
      if n [return pick dayblk num]
      if l [return pick dayblk (length? dayblk)]

      return dayblk

   ]
]

;**************************************************************************;
;*                  Generate a calendar for month & year                  *;
;*                                                                        *;
;* To generate a years worth (e.g.1999):                                  *;
;*                                                                        *;
;*    for n 1 12 1 [calendar n 1999]                                      *;
;*                                                                        *;
;**************************************************************************;

calendar: func [month year /col colwidth /OS] [

   use [smonth syear J2G temp n nday eom spacers] [

      J2G: false
      if not col [colwidth: 3]
      if colwidth < 2 [colwidth: 2]

      spacers: copies " " colwidth

      smonth: right/pad form month 2 "0"
      syear:  right/pad form year  4 "0"

      if (year < 1582) or ((year = 1582) and (month < 10)) [OS: true]
      else [
         if all [(not OS) (year = 1582) (month = 10)] [
            J2G: true
            OS: true
         ]
      ]

      if not OS [
         n: weekday/n make date! compress form [1 "/" month "/" year]
         eom: lastday month year
      ]
      else [
         n: OSweekday/n make date! compress form [1 "/" smonth "/" syear]
         eom: lastday/OS month year
      ]     

      print ""
      temp: pick names-of-months month
      temp: overlay temp right/pad form year 4 " " (colwidth * 7 + 9)
      print temp
      foreach day names-of-days [
         prin [left day colwidth " "]
      ]
      print ""
      for spacer 1 (n - 1) 1 [
         prin [spacers " "] 
      ]
      n: n - 1
      nday: 1

      if not J2G [      
         for wks 1 6 1 [
            for dates 1 (7 - n) 1 [
               prin [right/pad form nday colwidth " " " "]
               nday: nday + 1
               if nday > eom [break]
            ]
            n: 0
            print ""
            if nday > eom [break]
         ]
      ]
      else [
         for wks 1 6 1 [
            for dates 1 (7 - n) 1 [
               prin [right/pad form nday colwidth " " " "]
               nday: nday + 1
               if (nday = 5) [nday: 15]
               if nday > eom [break]
            ]
            n: 0
            print ""
            if nday > eom [break]
         ]

      ]
   ]
]
""
}

   "library"
   "datetime/dwdate.locale" 
   {REBOL [
   Title:   "Locale for dwdate.rlib"
   Author:  "Dick Whiting"
   Email:   dwhiting@europa.com
   Home:    http://www.europa.com/~dwhiting
   Date:    30-Dec-1998
   File:    %library/datetime/dwdate.locale
   Version: 1.0.0

   Requirements: {

      Is enabled by functions from dwstrings.rlib and dwsupport.rlib

      Set variable 'default-languages: ["lang1" "lang2" ...] in user.r 
      with your choices for languages, prior to performing dwsupport.rlib

      Clone this file (dwdate.locale) to dwdate.lang1 and edit to suit.
      Examples:

         default-languages: ["czech" "deutsch"]

         will use dwdate.czech if found, dwdate.deutsch next, or the internal
         values in dwdate.rlib, if neither is found.

   }

   Purpose: {

      Provide localization for dwdate.rlib

   }
]

;**************************************************************************;
;*  Modify for your language and then save as dwdate.xxxxx where xxxxx    *;
;*         is your language. Example: dwdate.czech, dwdate.deutsch        *;
;*      Follow directions in this header for enabling the localization    *;
;**************************************************************************;

names-of-days: [
      "Sunday" 
      "Monday" 
      "Tuesday" 
      "Wednesday" 
      "Thursday" 
      "Friday" 
      "Saturday"
]
 
abbrev-of-days: [
      "Sun" 
      "Mon" 
      "Tue" 
      "Wed" 
      "Thu" 
      "Fri" 
      "Sat"
]
 
names-of-months: [
      "January" 
      "February" 
      "March" 
      "April" 
      "May" 
      "June" 
      "July" 
      "August" 
      "September" 
      "October" 
      "November" 
      "December"
]

abbrev-of-months: [
      "Jan" 
      "Feb" 
      "Mar" 
      "Apr" 
      "May" 
      "Jun" 
      "Jul" 
      "Aug" 
      "Sep" 
      "Oct" 
      "Nov" 
      "Dec"
]


""}
]

;**************************************************************************;
;*    Function names in this package. Used for help and example files     *;
;**************************************************************************;
;*           Include file is created by build-fnames.r script             *;
;**************************************************************************;

func-names: 
["calendar" "datestr" "dwdate" "getdays" "greg2jul" "incrdate" "is-julian-date" "is-leap-year" "jul2greg" "lastday" "osweekday" "weekday"]


;**************************************************************************;
;*                         Help and Example files                         *;
;*                    File created using build-help.r                     *;
;**************************************************************************;

; START OF HELP INCLUDES


calendar.help: {
Usage: CALENDAR month year /col colwidth /OS

    Prints a Gregorian calendar for month of year. 

    /col sets the column width for the names of days (default 3)

    /OS prints a calendar for Julian month of year

    This function automatically converts from Gregorian to Julian calendars
    prior to October 1582. October 1582 shows the correction that occurred.

    To generate a years worth (e.g. 1999) use code like:

          "for n 1 12 1 [calendar n 1999]"
}
datestr.help: {
Usage: DATESTR date /euro /euro6 /julian /julian5 /sorted
                    /usa /usa6 /sep schar /fmt fmtstr 

    Returns a string with date formatted according to refinement.

    Formats (specify only one):
                                                          
    euro      -- ddmmyyyy i.e. 25/01/1999 for 25-Jan-1999
    euro6     -- ddmmyy   i.e.   25/01/99 for 25-Jan-1999 
    julian    -- yyyyddd  i.e.    1999025 for 25-Jan-1999 
    julian5   -- yyddd    i.e.      99025 for 25-Jan-1999
    sorted    -- yyyymmdd i.e.   19990125 for 25-Jan-1999 
    usa       -- mmddyyyy i.e. 01/25/1999 for 25-Jan-1999
    usa6      -- mmddyy   i.e.   01/25/99 for 25-Jan-1999 
      
    /sep schar can be used to change the separation character "/" to
    a different one, such as "-" for /euro /euro6 /usa /usa6

    Note: /sep " " will not work. You'll need to either use the /fmt
    refinement, or translate the result after doing datestr.

    /fmt fmtstr can be used to place date information into fmtstr.
    
    example: datestr/fmt now "On %MMM %dd, %yyyy, you said..."

    Codes      Replacement value

    %yyyy0     4-digit year padded with zeroes if necessary
    %yyyy      4-digit year without leading zeroes
    %yy0       2-digit year padded with zeroes if necessary
    %yy        2-digit year without leading zeroes

    %MMM       full name of month
    %mmm       abbreviated name of month
    %mm0       2-digit month padded with zeroes if necessary
    %mm        2-digit month without leading zeroes

    %DDD       full name of weekday
    %ddd       abbreviated name of weekday
    %dd0       2-digit day padded with zeroes if necessary
    %dd        2-digit day without leading zeroes

    %doy0      3-digit day of year padded with zeroes if necessary
    %doy       3-digit day of year without leading zeroes

    %time8     8-character hh:mm:ss padded with zeroes if necessary
    %time5     5-character hh:mm    padded with zeroes if necessary
    %time      x-character hh:mm:ss or hh:mm without leading zeroes
    %hr0       2-digit hour padded with zeroes if necessary
    %hr        2-digit hour without leading zeroes
    %min0      2-digit minute padded with zeroes if necessary
    %min       2-digit minute without leading zeroes
    %sec0      2-digit second padded with zeroes if necessary
    %sec       2-digit second without leading zeroes


    Notes: Partial seconds are removed. If the input date does not have
           a time portion, then time is set to 00:00:00
}
dwdate.help: {
Included DATETIME functions:

CALENDAR       print a calendar for a month and year (Gregorian or Julian)
DATESTR        returns a date in a variety of formats as a string
GETDAYS        return the nth weekday (e.g. 3rd Thursday) for month/year
GREG2JUL       convert a Gregorian date to a Julian date
INCRDATE       increment or decrement date or datetime values

IS-JULIAN-DATE test if a string describes a valid Julian date
JULIAN-DATE?   alias for is-julian-date             

IS-LEAP-YEAR   test if year is a valid Gregorian leap year
LEAP-YEAR?     alias for is-leap-year

JUL2GREG       convert a Julian date to a Gregorian date
LASTDAY        get last day for month and year (Gregorian or Julian)
OSWEEKDAY      get what day of the week for a Julian ("old-style") date
WEEKDAY        get what day of the week for a Gregorian date

Creates these global variables:

names-of-days     -- full names for days of the week
abbrev-of-days    -- 3 character abbreviations for days
names-of-months   -- full names for months of the year
abbrev-of-months  -- 3 character abbreviations for months
lengths-of-months -- number of days in each non-leap year month
base-sunday       -- date of a Sunday in the Gregorian calendar
OSbase-sunday     -- date of a Sunday in the Julian calendar
}
getdays.help: {
Usage: GETDAYS dayofweek month year /f /l /n nth /full

    Returns all days of month for dayofweek for month of year in a block.
    
    dayofweek may either be a string, i.e. "Tuesday" or the number of
    the weekday (Sunday = 1, Saturday = 7)

    /full returns the full date rather than just the day of the month

    /f will return only the FIRST dayofweek date either as a number or
       as a date! if /full is specified

    /l does the same, but returns the LAST dayofweek date

    /n returns the nth dayofweek date
      
    See also: WEEKDAY OSWEEKDAY INCRDATE
}
greg2jul.help: {
Usage: GREG2JUL date

    Returns the Julian equivalent for date. See warning. 

    NONE! is returned if the input is not in valid date! format.

    WARNING: since some Julian dates are invalid REBOL date! format,
    you need to test if you get back a date! or a string!

    If you get back a string! it is a Julian leapday that is illegal
    under Gregorian rules. 

    See also: JUL2GREG
}
incrdate.help: {
Usage: INCRDATE date /d dincr /w wincr /m mincr /y yincr
                     /hr hincr /min mnincr /sec sincr /none /keep /roll

    Returns a date incremented by refinement type and amount.

    Multiple refinements are allowed.

    /y   +/- number of years  (see /none /keep /roll)
    /m   +/- number of months (see /none /keep /roll)
    /w   +/- number of weeks  (1 week = 7 days)
    /d   +/- number of days
    /hr  +/- number of hours
    /min +/- number of minutes
    /sec +/- number of seconds

    When adding/subtracting years or months, the result can be ambiguous. 
    Examples:  

      29/2/2000 + 1 year  = 28/2/2001 OR  1/3/2001
      31/1/2001 + 1 month = 28/2/2001 ... 3/3/2001

    When one of these conditions can occur:

    /none -- says to return NONE! 
    /roll -- roll the extra days into the next month. This is the default.
    /keep -- keep the month constant and use its last day for the date.
    
    Note: 

      /keep and /roll may produce results that are NOT reversible, 
      i.e. adding a month to a date and then subtracting a month 
      from the result may not return the original value.

      As of Rebol v1.0.3.x: the maximum increment for days is 32K
                            the maximum increment for hr/min/sec is 99,999

      Order of application is largest time interval to smallest.

    See also: GETDAYS
}
is-julian-date.help: {
Usage: IS-JULIAN-DATE string

    Returns TRUE if string describes a valid Julian date; FALSE otherwise.

    String must have the format of "dd/mm/yyyy" (e.g. "2/29/1900")

    This function has an alias of JULIAN-DATE? predefined.

    See also: IS-LEAP-YEAR
}
is-leap-year.help: {
Usage: IS-LEAP-YEAR year

    Returns TRUE if year is a Gregorian leapyear, FALSE otherwise.

    This function has an alias of LEAP-YEAR? predefined.

    See also: IS-JULIAN-DATE
}
jul2greg.help: {
Usage: JUL2GREG date

    Returns the Gregorian equivalent for date in DATE! format.

    Input must be either in valid date! format or a string! in the form
    of "dd/mm/yyyy"

    The latter is the only way to handle Julian "extra" leapdays such as
    "29/2/1900" which is not a legal Gregorian date.


    NONE! is returned if the input is an invalid Julian date.



    See also: GREG2JUL
}
lastday.help: {
Usage: LASTDAY month year /OS

    Returns the number of the last day in month of year.

    /OS treats the input as a Julian date
}
osweekday.help: {
Usage: OSWEEKDAY date /n

    Returns the name of the day for Julian date as a string.

    date can be a valid date! value or a string in "dd/mm/yyyy" format.

    if date is a valid Julian leapday but NOT a valid Gregorian leapday,
    then the input MUST be in string format. (e.g. "29/2/1900")

    /n returns the number of the day (Sunday = 1, Saturday = 7)


    See also: WEEKDAY
}
weekday.help: {
Usage: WEEKDAY date /n

    Returns the name of the day for Gregorian date as a string.

    The input date must be in date! format.

    /n returns the number of the day (Sunday = 1, Saturday = 7)


    See also: OSWEEKDAY
}
calendar.example: {[
{calendar second now third now}
]
}
datestr.example: {[
{print [datestr/euro6 now]
print [datestr/euro6/sep now "-"]
print [datestr/fmt now "On %DDD, %MMM %dd, %yyyy at %time, you said..."]
}
]
}
getdays.example: {[
{print [getdays "Thursday" 1 2000]}
{print [getdays/full 5 1 2000]}
{print getdays/f 5 1 2000}
{print getdays/n 5 1 2000 3}
{print getdays/l 5 1 2000}
]
}
greg2jul.example: {[
{print greg2jul 15/10/1582}
{print greg2jul 13/03/1900}
{print greg2jul 22/02/1732}
]
}
incrdate.example: {[
{print [incrdate/y/m/w/d 30/9/1981 3 3 3 3]}
{print [incrdate/y/roll  29/2/2000 1]}
{print [incrdate/y/keep  29/2/2000 1]}
{print [incrdate/y/none  29/2/2000 1]}
]
}
is-julian-date.example: {[
{print is-julian-date "29/2/1900"}
{print is-julian-date "29/2/1901"}
{print julian-date? "01/1/2000"}
]
}
is-leap-year.example: {[
{print is-leap-year 2000}
{print is-leap-year 1900}
{print leap-year?   1901}
]
}
jul2greg.example: {[
{print jul2greg 05/10/1582}
{print jul2greg "29/02/1900"}
{print jul2greg 11/02/1732}
]
}
lastday.example: {[
{print lastday 2 1900}
{print lastday/OS 2 1900}
{print lastday 1 2000}
]
}
osweekday.example: {[
{print osweekday "29/2/1900"}
{print osweekday  28/2/1900}
{print osweekday/n 1/1/2001}
]
}
weekday.example: {[
{print weekday   29/2/2000}
{print weekday   28/2/1900}
{print weekday/n 1/1/2001}
]
}


; END OF HELP INCLUDES

;**************************************************************************;
;*                     FUNCTIONS USED WITHIN INSTALLER                    *;
;**************************************************************************;

;**************************************************************************;
;*         Create absolute paths to %library and %help directories        *;
;**************************************************************************;

ins-loadpaths: func [] [

   use [substr lastpos] [

      substr: func [str start cnt] [
         use [newstr] [
            newstr: head copy/part skip str (start - 1) cnt
            return newstr
         ]
      ]

      lastpos: func [pattern str] [
         use [found newstr] [
            found: false
            newstr: make string! str
            while [found? find newstr pattern] [
               newstr: next find newstr pattern 
               found: true
            ]
            if found [return (index? newstr) - 1]
            else [return 0]
         ]
      ]

      ins-rebol-path: make string! substr rebol/bootfile 1 ((lastpos "/" rebol/bootfile) - 1)

      if (exists? ins-rebol-path) [
         ins-rebol-path: head insert tail ins-rebol-path "/"
         ins-help-path:  make string! ins-rebol-path
         ins-help-path:  head insert tail ins-help-path "help/"
         ins-lib-path:   make string! ins-rebol-path
         ins-lib-path:   head insert tail ins-lib-path "library/"
         return true
      ] 
      else [return false]
   ]
]

;**************************************************************************;
;*                Get response from user. "Q" quits script                *;
;**************************************************************************;

ins-getans: func [] [

   ans: input
   if (length? ans) > 0 [ans: form first ans] else [ans: "Y"]
   if ans = "q" [halt] 
   return ans

]

;**************************************************************************;
;*                      Check that directories exist                      *;
;**************************************************************************;
ins-checkdirs: func [] [

   foreach [dtype dpath] required-dirs [
      if dtype = "library" [fullpath: make file! ins-lib-path]
      else [
         if dtype = "help" [fullpath: make file! ins-help-path]
      ]
      fullpath: head insert tail fullpath dpath
      while [not exists? fullpath] [
         print ["You need to create directory: " fullpath]
         prin "(Y/q): "
         ins-getans
      ]
   ]
]
;**************************************************************************;
;*                         INSTALLER STARTS HERE                          *;
;**************************************************************************;

;**************************************************************************;
;*    Let the user know what is going on. Give 'em a chance to stop it    *;
;*          Save each script and data file. Save the help files.          *;
;**************************************************************************;


if ((length? intromsg) > 0) [
   print intromsg
   prin "(Y/q): "
   ins-getans
]

if ((length? changesmsg) > 0) [
   print changesmsg
   prin "(Y/q): "
   ins-getans
]

if not ins-loadpaths [

   print "Unable to determine required path information!"
   print {

   Send bug report to:

      Dick Whiting <dwhiting@europa.com>
   }          
   print "Quitting install script..."
   halt
]

if ((length? byhandmsg) > 0) [
   print byhandmsg
   prin "(Y/q): "
   ins-getans
]

if ((length? required-dirs) > 0) [ins-checkdirs]

print ""
print "Starting Install.."
print ""
print {Enter "Q" at any prompt to Quit, "N" to skip to next step}
print ""


;**************************************************************************;
;*     Save each file, included as a string, to the specified filename    *;
;**************************************************************************;

foreach [dtype fpath file] files-to-install [

   if dtype = "library" [fullpath: make file! ins-lib-path]
   else [
      if dtype = "help" [fullpath: make file! ins-help-path]
   ]

   fullpath: head insert tail fullpath fpath

   print ""
   print ["Save" fullpath "?"]
   prin "(Y/n/q): "
   ins-getans

   if ans = "Y" [ 
      if (exists? fullpath) [
         print ""
         print [fullpath "already exists.. Overwrite it?"]
         prin "(Y/n/q): "
         ins-getans
      ]
      if ans = "Y" [
         write fullpath file
      ]
   ]
]

;**************************************************************************;
;*     Save a help-file for each function name in the func-names list     *;
;*                    Save an examples file, if present                   *;
;**************************************************************************;

print ""
print "Save help files? NO checking for already existing is done"
prin "(Y/n/q): "
ins-getans

if ans <> "n" [
   foreach [name] func-names [
   
      helpvar: make string! ".help"
      helpvar: head insert head helpvar name
   
      helpfile: make string! ins-help-path
      helpfile: insert tail helpfile script-help
      helpfile: insert tail helpfile "/"
      helpfile: insert tail helpfile name
      helpfile: head insert tail helpfile ".help"

      htest: make string! "value? '"
      htest: head insert tail htest helpvar

      if (do htest) [
         write make file! helpfile mold do helpvar
      ]

      examvar: make string! ".example"
      examvar: head insert head examvar name
   
      examfile: make string! ins-help-path
      examfile: insert tail examfile script-help
      examfile: insert tail examfile "/"
      examfile: insert tail examfile name
      examfile: head insert tail examfile ".example"
      
      etest: make string! "value? '"
      etest: head insert tail etest examvar

      if (do etest) [
         do form ["write make file!" examfile examvar]
      ]
   
   ]
]

;**************************************************************************;
;*                             Final messages                             *;
;**************************************************************************;

if ((length? finalmsg) > 0) [
   print Finalmsg
]

print ""
print "You can delete this install script now"
print ""
print "-- bug reports, comments, etc. to Dick Whiting <dwhiting@europa.com>"
print ""
print "done.."

print "Press any key to Quit"
ins-getans

;**************************************************************************;
;*           Quit so that variables, functions don't carry over.          *;
;**************************************************************************;

quit
