REBOL [
   Title:   "Install dwsupport.rlib pieces"
   Author:  "Dick Whiting"
   Email:   dwhiting@europa.com
   Home:    http://www.europa.com/~dwhiting
   Date:    18-Jan-1999
   File:    %install-dwsupport.r
   Version: 1.0.3

   Replaces: {Replaces previous versions of dwsupport.rlib}

   Usage: {
   
      Within the same directory that REBOL exists:

         Create a directory named "library/support/"
         Create a directory named "help/support/dwsupport/"

      Copy this file into your main REBOL directory

      Start REBOL interactively and enter:  

         do %install-dwsupport.r

      This script will do the following:

      1) Save file %library/support/dwsupport.rlib             (the functions)
      2) Save help files %help/support/dwsupport/*.help        (help files)
      3) Save example files %help/support/dwsupport/*.example  (example files)

      You can discard this script after the installation has been done.      
      
   }

   Purpose: {

      Handles installing dwsupport.rlib and its corequisites

   }

   Comment: {

      The "working part of the install script can be located by
      searching for the string "INSTALLER STARTS HERE" -- at bottom.

   }

]

;**************************************************************************;
;*                     Variables used during install                      *;
;**************************************************************************;

script-help: "support/dwsupport/"

;**************************************************************************;
;*                          Introduction message                          *;
;**************************************************************************;
;*                       Set Intromsg: "" to disable                      *;
;**************************************************************************;

Intromsg: {
   Title:   "A collection of support functions"
   Author:  "Dick Whiting"
   Email:   dwhiting@europa.com
   Home:    http://www.europa.com/~dwhiting
   Date:    18-Jan-1999
   File:    %library/support/dwsupport.rlib
   Version: 1.0.3

This script will do the following:

   1) Save file %library/support/dwsupport.rlib             (the functions)
   2) Save help files %help/support/dwsupport/*.help        (help files)
   3) Save example files %help/support/dwsupport/*.example  (example files)

Functions in dwsupport.rlib:

   libinfo   -- display information from an rlib header
   loadpaths -- create absolute paths to various directories
   loadlib   -- load and track order for libraries
   loadmsg   -- display messages if interactive, not silent, not script
   help      -- replaces REBOL help with EXHELP, calls old one if needed
   exhelp    -- dispalys help from external files, handles many forms
   inhelp    -- displays any help inlined within functions as a string
   userwords -- formats list of all currently defined user-context words
   
Press "Q" to quit or any other key to continue
}

;**************************************************************************;
;*                       Changes since last version                       *;
;**************************************************************************;
;*                     Set Changesmsg: "" to disable                      *;
;**************************************************************************;

Changesmsg: {

      Changes in 1.0.3:

      inhelp -- fixed to handle functions with AND without USE's
                added /called refinement for use with HELP
                a bit prettier (thanks -pekr-)

      exhelp -- added /inline refinement so internal help will be 
                checked after external help is not found

      help   -- now tries external, INTERNAL, and then default help


      help.help   -- updated for changes
      exhelp.help -- updated for changes
      inhelp.help -- updated for changes

}

;**************************************************************************;
;*             User has to do these by hand before installing             *;
;**************************************************************************;
;*                      Set ByHandMsg: "" to disable                      *;
;**************************************************************************;

ByHandMsg: {

Before continuing with this install script, you must create the following
directories within your main REBOL directory:

   library/support/
   help/support/dwsupport/

Press "Q" to quit or any other key to continue
}

;**************************************************************************;
;*                     Final message at end of install                    *;
;**************************************************************************;
;*                       Set FinalMsg: "" to disable                      *;
;**************************************************************************;

bldcmdline: func [] [
   use [newstr] [
      newstr: {do make file! "}
      insert tail newstr ins-lib-path
      insert tail newstr {support/dwsupport.rlib"}
      return head newstr
   ]
]

FinalMsg: [{

If you have not installed the dwstrings.rlib package, then you should do that
now. It provides functions that are REQUIRED by dwsupport.rlib.
When you have done that--

Add this line to your user.r script before others that use these functions:}

newline
newline 
"      " bldcmdline 
newline


{

You may also want to make use of the "localization" facility by placing
a line such as the following in your user.r file:

           default-languages: ["czech" "deutsch"]
}
]

;**************************************************************************;
;*                      List of required directories                      *;
;*             List is pairs with location and subdir needed              *;
;**************************************************************************;
;*                    Set required-dirs: [] to disable                    *;
;**************************************************************************;

required-dirs: [

   "library" "support" 
   "help"    "support/dwsupport"
]

;**************************************************************************;
;*                          Files to be installed                         *;
;* List is triad with path type, file path, and file included as a string *;
;**************************************************************************;

files-to-install: [ 

   "library" 
   "support/dwsupport.rlib" 
   {REBOL [
   Title:   "A collection of support functions"
   Author:  "Dick Whiting"
   Email:   dwhiting@europa.com
   Home:    http://www.europa.com/~dwhiting
   Date:    15-Jan-1999
   File:    %library/support/dwsupport.rlib
   Version: 1.0.3

   Changes: {

      Changes in 1.0.3:

      inhelp -- fixed to handle functions with AND without USE's
                added /called refinement for use with HELP
                a bit prettier (thanks -pekr-)

      exhelp -- added /inline refinement so internal help will be 
                checked after external help is not found

      help   -- now tries external, INTERNAL, and then default help


      help.help   -- updated for changes
      exhelp.help -- updated for changes
      inhelp.help -- updated for changes
   }

   Requirements: {

      You need to create a directory tree within the main REBOL directory
      as follows:

         Library/          -- directory for function libraries 
         Library/strings/  -- place string libraries here
         Library/support/  -- dwsupport.rlib goes in this directory
         Library/...       -- add others as the opportunity arises

         Help/             -- directory for help directories
         Help/strings/     
         Help/strings/dwstrings/ -- dwstrings help files here
         Help/support/dwsupport/ -- dwsupport help files here
         ... etc...        -- add others as the opportunity arises
   }

   Uses: {

      These functions from the dwstrings.rlib package:
      
         COMPRESS 
         OVERLAY
         POS 
         SCANSTR
         SUBSTR
   }

   Purpose: {

      Provide a variety of support functions for REBOL, including tracking 
      installed libraries, external and internal help, displaying all current
      words in the user-context, etc.
 
   }

   Functions: {

      libinfo   -- display information from an rlib header
      loadpaths -- create absolute paths to various directories
      loadlib   -- load and track order for libraries
      loadmsg   -- display messages if interactive, not silent, not script
      help      -- replaces REBOL help with EXHELP, calls old one if needed
      exhelp    -- dispalys help from external files, handles many forms
      inhelp    -- displays any help inlined within functions as a string
      userwords -- formats list of all currently defined user-context words
   }

   Variables: {

      Creates these global variables:
      
      rebol-path  -- absolute path to the main rebol directory
      lib-path    -- absolute path to %library
      help-path   -- absoulute path tto %help
      libs-loaded -- libraries loaded with the loadlib function
   }

   Comment: {

      This is my attempt to implement various suggestions for REBOL
      organization. Please make comments on the REBOL user-list as to
      whether this scheme works, what needs to be modified, deleted, etc.

      I arbitrarilly decided to give function packages a suffix of ".rlib"
      It seemed that this would be an easy way to distinguish between single
      function scripts and ones with a whole collection.

      As of the initial release date, this may seem rather overblown as
      the only "libraries" of functions are dwsupport and dwstrings. The
      need for some scheme such as this will become more important as all
      those math libs, statistic libs, surveyor libs, etc. get posted.

      That was an unsubtle hint;)

   }

   Uninstall: [
   
      "library" "support/dwsupport.rlib"
      "help"    "support/dwsupport"
      "user"    "references to dwsupport"

   ]
]

;**************************************************************************;
;*         Create absolute paths to %library and %help directories        *;
;**************************************************************************;

loadpaths: func [] [

   use [substr lastpos allfound bootfile] [

      substr: func [str start cnt] [
         use [newstr] [
            newstr: head copy/part skip str (start - 1) cnt
            return newstr
         ]
      ]

      lastpos: func [pattern str] [
         use [found newstr] [
            found: false
            newstr: make string! str
            while [found? find newstr pattern] [
               newstr: next find newstr pattern 
               found: true
            ]
            if found [return (index? newstr) - 1]
            else [return 0]
         ]
      ]

      allfound: true

      ;*******************************************;
      ;* Handle v1.0.3.x rebol/bootfile problems *;
      ;*******************************************;

      if block? rebol/bootfile [
         bootfile: first rebol/bootfile
      ]
      else [
         bootfile: rebol/bootfile
      ]

      rebol-path: make string! substr bootfile 1 ((lastpos "/" bootfile) - 1)

      if (not exists? rebol-path) [allfound: false]
      else [
         rebol-path: head insert tail rebol-path "/"
         help-path:  make string! rebol-path
         help-path:  head insert tail help-path "help"
         if (not exists? help-path) [allfound: false]
         else [
            help-path:  head insert tail help-path "/"
            lib-path:   make string! rebol-path
            lib-path:   head insert tail lib-path "library"
            if (not exists? lib-path) [allfound: false]
            else [
               lib-path:   head insert tail lib-path "/"
            ]
         ]
      ]
      return allfound
   ]
]

;**************************************************************************;
;* Only prints if in INTERACTIVE mode, without --script set, not SILENT   *;
;**************************************************************************;

loadmsg: func [str] [

  if all [(not rebol/silent) 
          (none? rebol/script)
          (rebol/interactive)
         ] [
     print [str]
  ]   

]

;**************************************************************************;
;* Create full path to lib, perform a "do" on the lib, remove its name    *;
;* from the libs-loaded block if present, and insert it at the head to    *;
;* allow proper order of searches for external help function.             *;
;**************************************************************************;

loadlib: func [name] [

   use [tmpname tmpfile locname locfile] [
      tmpname: make string! lib-path 
      tmpname: insert tail tmpname name
      locname: make string! head tmpname
      tmpname: head insert tail tmpname ".rlib"
      tmpfile: make file! tmpname
      if (exists? tmpfile) [

         loadmsg form ["Loading" tmpname "..."]
         do tmpfile   

         if (value? 'default-languages) [
            foreach lang default-languages [
               locfile: make string! locname
               locfile: insert tail locfile "."
               locfile: head insert tail locfile lang
               if (exists? make file! locfile) [
                  do make file! locfile
                  break
               ]
            ]
         ]

         if not value? 'libs-loaded [
            libs-loaded: make block! ["support/dwsupport"]
         ]

         for num 1 (length? libs-loaded) 1 [
            if (pick libs-loaded num) = name [
               remove/part skip head libs-loaded (num - 1) 1
               break
            ]
         ]
         libs-loaded: head insert head libs-loaded name
         exit
      ]
      else [print [tmpfile "not found"]]
   ]
]


;**************************************************************************;
;*                    Display info from an rlib header                    *;
;**************************************************************************;
;*               Uses scanstr & compress from dwstrings.rlib              *;
;**************************************************************************;

libinfo: func ['name 'field] [

   use [tmpfile libhead tinfo helper] [

      helper: func [tmpfile field] [
         libhead: first load tmpfile
         if (form field) = "?" [
            tinfo: make block! 0
            for n 1 length? libhead 2 [
               tinfo: insert tail tinfo pick libhead n
            ]
            return head tinfo
         ]
         else [
            if ((form field) = "") or ((form field) = "all") [
               return libhead
            ]
            else [
               return do compress form ["libhead/" field]
            ]
         ]
      ]

      if path? name [
         tmpfile: make file! compress form [lib-path name ".rlib"]
         if exists? tmpfile [return helper tmpfile field]
      ]

      foreach lname libs-loaded [
         if (scanstr/last lname "/") = (form name) [
            tmpfile: make file! compress form [lib-path lname ".rlib"]
            if exists? tmpfile [return helper tmpfile field]
         ]
      ]
      return none!
   ]
]

;**************************************************************************;
;*  The order of looking for help information is:                         *;
;*                                                                        *;
;*  1) See if %library/name/ is a directory, if so list it                *;
;*  2) See if %help/path.help exists, if so display it                    *;
;*  3) See if name.help can be found within help directories for libs     *;
;*     that were loaded with the "loadlib" function. If so, display it.   *;
;*     This insures that the most recent function with a name has ITS     *;
;*     helpfile displayed.                                                *;
;*  4) Display error message or pass name to REBOL help command           *;
;**************************************************************************;

exhelp: func ['name /e /a /f /old /inline] [

   use [ldir rname langdir hfile efile example locpath pblock pname] [


;**************************************************************************;
;*                    list the main Library directory                     *;
;**************************************************************************;

      if (name = "") or ((form name) = "?") [
         ldir: load make file! lib-path
         foreach fname ldir [
            if any [((pos "." fname) = 0) 
                    ((scanstr/last fname ".") = "r") 
                    ((scanstr/last fname ".") = "rlib")] [print fname]
         ]
         exit
      ]

;**************************************************************************;
;*                    List secondary Library directory                    *;
;**************************************************************************;

      hfile: make file! compress form [lib-path name]

      if (exists? hfile) [

;;-- !!! The following line appends a slash since LOAD dies without it
;;-- !!! However, EXISTS? dies WITH it in the Wxx environment :(

         hfile: head insert tail hfile "/"

         ldir: load hfile
         foreach fname ldir [
            if any [((pos "." fname) = 0) 
                    ((scanstr/last fname ".") = "r") 
                    ((scanstr/last fname ".") = "rlib")] [print fname]
         ]
         exit ;-- done printing directory
      ]

;**************************************************************************;
;*               Build a list of localization partial paths               *;
;**************************************************************************;

      locpath: make block! 0

      if (value? 'default-languages) [
         foreach lang default-languages [
            insert tail locpath lang
         ]
      ]
      locpath: head insert tail locpath ""

;**************************************************************************;
;*        List path with command, e.g. "strings/dwstrings/compare"        *;
;**************************************************************************;

      pblock: make block! 0
      if path? name [
         pblock: scanstr form name "/"
         rname: last pblock 
      ]

      if (path? name) and ((length? pblock) > 2) [
         pname: make string! 0
         pname: for n 1  ((length? pblock)- 1) 1 [
            insert tail pname compress form [pick pblock n "/"]
         ]
         pname: head clear past tail pname

         foreach loc locpath [
            if loc <> "" [
               langdir: make file! compress form [help-path pname "/" loc]
               hfile: make file! compress form [help-path pname "/" loc "/" rname ".help"]
               efile: make file! compress form [help-path pname "/" loc "/" rname ".example"]
            ]
            else [
               langdir: make file! compress form [help-path pname]
               hfile: make file! compress form [help-path pname "/" rname ".help"]
               efile: make file! compress form [help-path pname "/" rname ".example"]
            ]

            if (exists? langdir) [
               if (exists? hfile)  [
                 if f [
                    print ""
                    print [hfile] 
                 ]
                 if not e [print load hfile]
                 if ((e or a) and (exists? efile)) [
                    example: load efile
                    print ""
                    print ["Examples:" newline]
                    foreach [str] example [
                       print [str]
                       do str
                       print ""
                    ]
                 ]
                 exit ;-- done printing help information
              ]
            ]
         ]
      ]

;**************************************************************************;
;*   Show rlib help file, e.g. "dwstrings.help" for "strings/dwstrings"   *;
;**************************************************************************;

      if (path? name) [
         foreach loc locpath [
            if loc <> "" [
               langdir: make file! compress form [help-path name "/" loc]
               hfile: make file! compress form [help-path name "/" loc "/" rname ".help"]
            ]
            else [
               langdir: make file! compress form [help-path name]
               hfile: make file! compress form [help-path name "/" rname ".help"]
            ]

            if (exists? langdir) [
               if (exists? hfile)  [
                 if f [print hfile newline]
                 print load hfile
                 exit  ;-- quit after displaying file
              ]
           ]
         ]
         print ["No helpfile found for" name]
         exit ;-- printed help or error msg for path type help
      ]

;**************************************************************************;
;*                Look for function name in all libs-loaded               *;
;**************************************************************************;

      foreach lname libs-loaded [
         foreach loc locpath [
            if loc <> "" [
               langdir: make file! compress form [help-path lname "/" loc]
               hfile: make file! compress form [help-path lname "/" loc "/" name ".help"]
               efile: make file! compress form [help-path lname "/" loc "/" name ".example"]
            ]
            else [
               langdir: make file! compress form [help-path lname]
               hfile: make file! compress form [help-path lname "/" name ".help"]
               efile: make file! compress form [help-path lname "/" name ".example"]
            ]

            if (exists? langdir) [
               if (exists? hfile)  [
                 if f [
                    print ""
                    print [hfile]
                 ]
                 if not e [print load hfile]
                 if ((e or a) and (exists? efile)) [
                    example: load efile
                    print ""
                    print ["Examples:" newline]
                    foreach [str] example [
                       print [str]
                       do str
                       print ""
                    ]
                 ]
                 exit ;-- exit when done printing help and/or example
               ]
            ]
         ]
      ]


;**************************************************************************;
;*                      Attempt to find inlined help                      *;
;**************************************************************************;

      if inline [
         if do form ["inhelp/called" name] [exit]
      ]

;**************************************************************************;
;*                 Was unable to find ANY help for 'name..                *;
;**************************************************************************;

      if old [
         do reduce ['old-help :name]
      ]
      else [
         print ["No help found for" name]
      ]

   ]
]

;**************************************************************************;
;*  Call EXHELP with the /old refinement. If no help is found otherwise,  *;
;*  attempts to use the standard REBOL help command (saved as "old-help") *;
;**************************************************************************;

if not value? 'old-help [old-help: :help]

help: func ['name /e /a /f /old] [

   if old [
      do reduce ['old-help name]
      exit
   ]

   use [helpstr] [
      helpstr: make string! "EXHELP"
      if e [insert tail helpstr "/e"]
      if a [insert tail helpstr "/a"]
      if f [insert tail helpstr "/f"]
      insert tail helpstr "/old"
      if not old [insert tail helpstr "/inline"]
      insert tail helpstr " "
      if name = "" [
         helpstr: head insert tail helpstr form [{""}]
      ]
      else [
         helpstr: head insert tail helpstr form name
      ]
      do helpstr
   ]
]

;**************************************************************************;
;*        Display any help inlined as a string within the function.       *;
;**************************************************************************;
;*    This function uses SUBSTR and POS from the dwstrings.rlib package   *;
;**************************************************************************;
inhelp: func ['name /called] [

   use [newstr shelp ehelp] [
      shelp: make integer! 0
      ehelp: make integer! 0
      if value? name [
         if function? get name [
            newstr: mold get name
            loop 3 [newstr: find next newstr "["]
            if (substr newstr 2 3) = "use" [
               loop 2 [newstr: find next newstr "["]
            ]
            if ((second newstr) = (make char! {"})) [
               shelp: (index? newstr) + 1
               ehelp: pos/after {"} newstr shelp
            ]
         ]
         else [
            if called [return false]
            print [name "is not a function"]
            exit
         ]
      ]
      else [
         if called [return false]
         print ["No value assigned to" name] 
         exit
      ]

      if (shelp > 0) and (ehelp > 0) [
         print [substr head newstr (shelp + 1) (ehelp - shelp - 1)]
         if called [return true]
      ]
      else [
         if called [return false]
         print ["No inlined information for" name]
         exit
      ]
   ]
]

;**************************************************************************;
;*        Function to list all currently defined user-context words       *;
;**************************************************************************;
;*       This function uses OVERLAY from the dwstrings.rlib package       *;
;**************************************************************************;

userwords: func [/cols cnt /width wide /unsorted] [
   use [a ctr line maxlen] [
      a: context-words user-context
      maxlen: 18
      foreach word a [if (length? form word) > maxlen [maxlen: length? form word]]
      if not unsorted [a: sort a]
      if not width [wide: maxlen + 2]
      if not cols [cnt: 5]
      ctr: make integer! 0
      line: make string! " "
      print line
      forall a [
         line: overlay line form first a ((ctr * wide) + 1)
         ctr: ctr + 1
         if ctr = cnt [
            print line
            ctr: 0
            line: " "
         ]
      ]
      print line
   ]
]

;**************************************************************************;
;*                            Create the paths                            *;
;**************************************************************************;

if not loadpaths [
   print "Not all needed directories exist"
]
else [
   if not value? 'libs-loaded [
      libs-loaded: make block! ["support/dwsupport"]
      loadmsg "Loading %library/support/dwsupport..."
   ]
]

}

]

;**************************************************************************;
;*    Function names in this package. Used for help and example files     *;
;**************************************************************************;
;*           Include file is created by build-fnames.r script             *;
;**************************************************************************;

func-names: 
["dwsupport" "exhelp" "help" "inhelp" "libinfo" "loadlib" "loadmsg" "loadpaths" "userwords"]


;**************************************************************************;
;*                         Help and Example files                         *;
;*                    File created using build-help.r                     *;
;**************************************************************************;

; START OF HELP INCLUDES


dwsupport.help: {
Included SUPPORT functions:

EXHELP    displays %library directories, external help files.

HELP      uses EXHELP to display external help files. Calls original help
          command if unable to locate information.

INHELP    displays help that is included in function definitions as a string.

LIBINFO   display information from an rlib header

LOADMSG   used to display messages to the REBOL interactive session.

LOADLIB   inserts a library name in block "libs-loaded" and does a "DO" on it.

LOADPATHS sets global variables "rebol-path" "libs-path" "help-path"

USERWORDS formats list of all currently defined user-context words
}
exhelp.help: {
Usage: EXHELP name /e /a /f /old /inline

    Provides a variety of help function support.

    EXHELP ""   -- displays directory for %library

    EXHELP name  -- displays directory for %library/name if it exists
                 -- searches for name.help for the currently loaded libraries
                 -- invokes the original REBOL help command if nothing found

    EXHELP name1/name2 -- directory list for %library/name1/name2 if it exists
                       -- displays %help/name1/name2.help if it exists

    /e -- display only the name.example file, ignored for directories
    /a -- display both name.help and name.example, ignored for directories
    /f -- display the full path for the help file
    /old -- used to toggle displaying "not found" message or invoking the 
            original REBOL help command against name. Ignored when name is
            specified as a path, i.e. "support/dwsupport/exhelp"

    /inline -- attempt to find inlined help, using inhelp, if external help
               file is not found.

    These use the paths variables created by the LOADPATHS function.

    Language localization is provided by:

      1) add a line to your user.r similar to this one with your preferred
         languages listed in order:

            default-languages: ["czech" "deutsch"]

      2) create a directory within the appropriate %help directory and place
         the localized help files in it. Example:

            %help/strings/dwstrings/czech/*.help
            %help/strings/dwstrings/czech/*.example

      If there are no localized versions available, then the one in the main
      help path will be used. Example:

            if %help/strings/dwstrings/czech/*.help doesn't exist, then
               %help/strings/dwstrings/*.help will be used.

    See also: [help inhelp loadpaths loadlib]
}
help.help: {
Usage: HELP name /e /a /f /old

    OLD-HELP name

    This pair changes the original HELP command to OLD-HELP.
    EXHELP is called to display external help information and if it
    isn't successful, looks for internal help, and finally the original 
    REBOL help command is attempted.


    HELP ""   -- displays directory for %library
    HELP ?    -- displays directory for %library

    HELP name  -- displays directory for %library/name if it exists
               -- searches for name.help for the currently loaded libraries
               -- invokes the original REBOL help command if nothing found

    HELP name1/name2 -- directory list for %library/name1/name2/ if it exists
                     -- displays %help/name1/name2.help if it exists

    /e -- display only the name.example file, ignored for directories
    /a -- display both name.help and name.example, ignored for directories
    /f -- display the full path for the help file
    /old -- will use the original help command without trying EXHELP
    /inline -- check for inlined help if external help not found

    These use the paths variables created by the LOADPATHS function.

    See also: [exhelp inhelp loadpaths loadlib]
}
inhelp.help: {
Usage: INHELP name /called

    Returns any inlined documentation for the function
    with the specified name. The inlined help must immediately follow
    the left bracket that defines the function and NOT contain any 
    double quote marks, i.e:

    A: func [args] [ "This is some helpful information"

      ... actual function code


    /called refinement suppresses failure messages and returns FALSE.

    ]
}
libinfo.help: {
Usage: LIBINFO name (header-word | ? | ALL)

   Displays information from an rlib header. 

   Specify ?   for header-word to get a block with all valid header-words.
   Specify ALL for header-word to disply the entire header.

   Typical vaild header-words:

        author 
        comment
        date
        email
        functions
        home
        purpose
        requires
        uninstall
        variables
        version
}
loadlib.help: {
Usage: LOADLIB str

   Looks in the libs-loaded global variable block. If it finds str, then
   it deletes it; then inserts str at the head of libs-loaded. 

   It then takes the str and appends it to the libs-path global variable,
   appends ".rlib" to it and performs a "DO" on the resulting full pathname.

   example: loadlib "strings/dwstrings"
      
   might result in: 

      do make file! "E7:/REBOL/Library/strings/dwstrings.rlib"

}
loadmsg.help: {
Usage: LOADMSG str

   If REBOL is running interactively, not SILENT, and not --SCRIPT then
   the str argument is printed. If any of the conditions is not true, then
   it just returns without doing anything. 

   Use it to suppress messages that you might not wish to display under
   script, silent, or non-interactive conditions.
}
loadpaths.help: {
Usage: LOADPATHS

   Constructs three global variables with absolute pathing using the 
   REBOL/BOOTFILE value.

   rebol-path: %
   lib-path:   %library
   help-path:  %help


   These are used by other dwsupport.rlib functions.
}
userwords.help: {
Usage: USERWORDS /cols cnt /width wide /unsorted

    Requires: OVERLAY function from dwstrings.r 

    Lists all words that are currently defined in the user-context.
    Output is formatted by default into 5 columns 20 characters wide
    and sorted alphabetically.

    /cols  cnt    sets the number of columns
    /width wide   sets the width of each column
    /unsorted     doesn't sort the words prior to displaying

    Note: Doesn't show the infix operators such as * ** //

          Use echo to write it to a file.
          >> echo %t:/userwords.text
}
libinfo.example: {[
{print [libinfo dwsupport ?]}
{print [libinfo support/dwsupport author]}
{print [libinfo dwsupport version]}
]
}
loadpaths.example: {[
{print rebol-path}
{print lib-path}
{print help-path}
]}


; END OF HELP INCLUDES

;**************************************************************************;
;*                     FUNCTIONS USED WITHIN INSTALLER                    *;
;**************************************************************************;

;**************************************************************************;
;*         Create absolute paths to %library and %help directories        *;
;**************************************************************************;

ins-loadpaths: func [] [

   use [substr lastpos bootfile] [

      substr: func [str start cnt] [
         use [newstr] [
            newstr: head copy/part skip str (start - 1) cnt
            return newstr
         ]
      ]

      lastpos: func [pattern str] [
         use [found newstr] [
            found: false
            newstr: make string! str
            while [found? find newstr pattern] [
               newstr: next find newstr pattern 
               found: true
            ]
            if found [return (index? newstr) - 1]
            else [return 0]
         ]
      ]

      ;*******************************************;
      ;* Handle v1.0.3.x rebol/bootfile problems *;
      ;*******************************************;

      if block? rebol/bootfile [
         bootfile: first rebol/bootfile
      ]
      else [
         bootfile: rebol/bootfile
      ]

      ins-rebol-path: make string! substr bootfile 1 ((lastpos "/" bootfile) - 1)

      if (exists? ins-rebol-path) [
         ins-rebol-path: head insert tail ins-rebol-path "/"
         ins-help-path:  make string! ins-rebol-path
         ins-help-path:  head insert tail ins-help-path "help/"
         ins-lib-path:   make string! ins-rebol-path
         ins-lib-path:   head insert tail ins-lib-path "library/"
         return true
      ] 
      else [return false]
   ]
]

;**************************************************************************;
;*                Get response from user. "Q" quits script                *;
;**************************************************************************;

ins-getans: func [] [

   ans: input
   if (length? ans) > 0 [ans: form first ans] else [ans: "y"]
   if ans = "q" [halt] 
   return ans

]

;**************************************************************************;
;*                      Check that directories exist                      *;
;**************************************************************************;
ins-checkdirs: func [] [

   foreach [dtype dpath] required-dirs [
      if dtype = "library" [fullpath: make file! ins-lib-path]
      else [
         if dtype = "help" [fullpath: make file! ins-help-path]
      ]
      fullpath: head insert tail fullpath dpath
      while [not exists? fullpath] [
         print ["You need to create directory: " fullpath]
         prin "(Y/q): "
         ins-getans
      ]
   ]
]
;**************************************************************************;
;*                         INSTALLER STARTS HERE                          *;
;**************************************************************************;

;**************************************************************************;
;*    Let the user know what is going on. Give 'em a chance to stop it    *;
;*          Save each script and data file. Save the help files.          *;
;**************************************************************************;


if ((length? intromsg) > 0) [
   print intromsg
   prin "(Y/q): "
   ins-getans
]

if ((length? changesmsg) > 0) [
   print changesmsg
   prin "(Y/q): "
   ins-getans
]

if not ins-loadpaths [

   print "Unable to determine required path information!"
   print {

   Send bug report to:

      Dick Whiting <dwhiting@europa.com>
   }          
   print "Quitting install script..."
   halt
]

if ((length? byhandmsg) > 0) [
   print byhandmsg
   prin "(Y/q): "
   ins-getans
]

if ((length? required-dirs) > 0) [ins-checkdirs]

print ""
print "Starting Install.."
print ""
print {Enter "Q" at any prompt to Quit, "N" to skip to next step}
print ""


;**************************************************************************;
;*     Save each file, included as a string, to the specified filename    *;
;**************************************************************************;

foreach [dtype fpath file] files-to-install [

   if dtype = "library" [fullpath: make file! ins-lib-path]
   else [
      if dtype = "help" [fullpath: make file! ins-help-path]
   ]

   fullpath: head insert tail fullpath fpath

   print ""
   print ["Save" fullpath "?"]
   prin "(Y/n/q): "
   ins-getans

   if ans = "Y" [ 
      if (exists? fullpath) [
         print ""
         print [fullpath "already exists.. Overwrite it?"]
         prin "(Y/n/q): "
         ins-getans
      ]
      if ans = "Y" [
         write fullpath file
      ]
   ]
]

;**************************************************************************;
;*     Save a help-file for each function name in the func-names list     *;
;*                    Save an examples file, if present                   *;
;**************************************************************************;

print ""
print "Save help files? NO checking for already existing is done"
prin "(Y/n/q): "
ins-getans

if ans <> "n" [
   foreach [name] func-names [
   
      helpvar: make string! ".help"
      helpvar: head insert head helpvar name
   
      helpfile: make string! ins-help-path
      helpfile: insert tail helpfile script-help
      helpfile: insert tail helpfile "/"
      helpfile: insert tail helpfile name
      helpfile: head insert tail helpfile ".help"

      htest: make string! "value? '"
      htest: head insert tail htest helpvar

      if (do htest) [
         write make file! helpfile mold do helpvar
      ]

      examvar: make string! ".example"
      examvar: head insert head examvar name
   
      examfile: make string! ins-help-path
      examfile: insert tail examfile script-help
      examfile: insert tail examfile "/"
      examfile: insert tail examfile name
      examfile: head insert tail examfile ".example"
      
      etest: make string! "value? '"
      etest: head insert tail etest examvar

      if (do etest) [
         do form ["write make file!" examfile examvar]
      ]
   
   ]
]

;**************************************************************************;
;*                             Final messages                             *;
;**************************************************************************;

if ((length? finalmsg) > 0) [
   print Finalmsg
]

print ""
print "You can delete this install script now"
print ""
print "-- bug reports, comments, etc. to Dick Whiting <dwhiting@europa.com>"
print ""
print "done.."

print "Press any key to Quit"
ins-getans

;**************************************************************************;
;*           Quit so that variables, functions don't carry over.          *;
;**************************************************************************;

quit
