REBOL [
   Title:   "Install dwstrings.rlib pieces"
   Author:  "Dick Whiting"
   Email:   dwhiting@europa.com
   Home:    http://www.europa.com/~dwhiting
   Date:    18-Jan-1999
   File:    %install-dwstrings.r
   Version: 1.1.3
   Replaces: "dwstrings.rlib v1.1.2 by Dick Whiting"

   Usage: {
   
      Within the same directory that REBOL exists:

         Create a directory named "library/strings/"
         Create a directory named "help/strings/dwstrings/"

      Copy this file into your main REBOL directory

      Start REBOL interactively and enter:  

         do %install-dwstrings.r

      This script will do the following:

      1) Save file %library/strings/D01to255.dat          (ascii values 1-255)
      2) Save file %library/strings/dwstrings.rlib             (the functions)
      3) Save help files %help/strings/dwstrings/*.help        (help files)
      4) Save example files %help/strings/dwstrings/*.example  (example files)

      You can discard this script after the installation has been done.      
      
   }

   Purpose: {

      Handles installing dwstrings.rlib and its corequisites

   }

   Comment: {

      The "working part of the install script can be located by
      searching for the string "INSTALLER STARTS HERE" -- at bottom.

   }

]

;**************************************************************************;
;*                     Variables used during install                      *;
;**************************************************************************;
script-help: "strings/dwstrings/"

;**************************************************************************;
;*                          Introduction message                          *;
;**************************************************************************;
;*                       Set Intromsg: "" to disable                      *;
;**************************************************************************;

Intromsg: {

This script installs all of the pieces for:

   Title:   "A collection of (A)Rexx-like string functions"
   Author:  "Dick Whiting"
   Email:   dwhiting@europa.com
   Home:    http://www.europa.com/~dwhiting
   Date:    18-Jan-1999
   File:    %library/strings/dwstrings.rlib
   Version: 1.1.3


This package is meant to replace rexxsfuncs.r. If you have that version, 
then you may delete it after installing this one.

Also replaces previous versions of dwstrings.rlib.

This script will do the following:

   1) Save file %library/strings/D01to255.dat          (ascii values 1-255)
   2) Save file %library/strings/dwstrings.rlib             (the functions)
   3) Save help files %help/strings/dwstrings/*.help        (help files)
   4) Save example files %help/strings/dwstrings/*.example  (example files)

Press "Q" to quit or any other key to continue
}

;**************************************************************************;
;*                       Changes since last version                       *;
;**************************************************************************;
;*                     Set Changesmsg: "" to disable                      *;
;**************************************************************************;

Changesmsg: {

Changes in dwstrings.rlib v1.1.3

      A code cleanup release. Improvements are on an A4000/040/25.
      Numbers were generated by looping execution of the old version
      and the new version for the same number of times and are for
      comparison puposes only.
      

      function   --  new-version    old-version

      blockwords --    0:00:57        0:01:03
      compress   --    0:00:23        0:00:31 
      copies     --    0:00:04        0:02:00  (copies "XXXX" 1000) * 5
      lastpos    --    0:00:14        0:00:25
      pos        --    0:00:13        0:00:16
      replstr    --    0:01:15        0:03:09
      reverse    --    0:00:24        0:00:30
      scanstr    --    0:00:24        0:00:28


      wraptxt    --  added a function to insert newlines in a string.

Press "Q" to quit or any other key to continue
}

;**************************************************************************;
;*             User has to do these by hand before installing             *;
;**************************************************************************;
;*                      Set ByHandMsg: "" to disable                      *;
;**************************************************************************;

ByHandMsg: {

Before continuing with this install script, you must create the following
directories within your main REBOL directory:

   library/strings/
   help/strings/dwstrings/

Press "Q" to quit or any other key to continue
}

;**************************************************************************;
;*                     Final message at end of install                    *;
;**************************************************************************;
;*                       Set FinalMsg: "" to disable                      *;
;**************************************************************************;

FinalMsg: {

If you have not installed the dwsupport.rlib package, then you should do that
now. It provides the functions supporting external help and example files, 
and the tracking of loaded libraries. When you have done that--

Add the following line to your user.r script, if you want these functions
available each time you start REBOL:

   loadlib "strings/dwstrings"

}

;**************************************************************************;
;*                      List of required directories                      *;
;*             List is pairs with location and subdir needed              *;
;**************************************************************************;
;*                    Set required-dirs: [] to disable                    *;
;**************************************************************************;

required-dirs: [

   "library" "strings" 
   "help"    "strings/dwstrings"
]

;**************************************************************************;
;*                          Files to be installed                         *;
;* List is triad with path type, file path, and file included as a string *;
;**************************************************************************;

files-to-install: [ 

   "library" 
   "strings/dwstrings.rlib" 
   {REBOL [
   Title:   "A collection of (A)Rexx-like string functions"
   Author:  "Dick Whiting"
   Email:   dwhiting@europa.com
   Home:    http://www.europa.com/~dwhiting
   Date:    15-Jan-1999
   File:    %library/strings/dwstrings.rlib
   Version: 1.1.3

   Changes: {

      A code cleanup release. Improvements are on an A4000/040/25.
      Numbers were generated by looping execution of the old version
      and the new version for the same number of times and are for
      comparison puposes only.
      

      function   --  new-version    old-version

      blockwords --    0:00:57        0:01:03
      compress   --    0:00:23        0:00:31 
      copies     --    0:00:04        0:02:00  (copies "XXXX" 1000) * 5
      lastpos    --    0:00:14        0:00:25
      pos        --    0:00:13        0:00:16
      replstr    --    0:01:15        0:03:09
      reverse    --    0:00:24        0:00:30
      scanstr    --    0:00:24        0:00:28


      wraptxt    --  added a function to insert newlines in a string.

   }

   Requirements: {

      The D01to255.dat file is read using information provided by the 
      dwsupport functions and needs to reside in %library/strings/

      If you decide not to use dwsupport, then modify the READ statement
      for D01to255.dat as needed.

      The help files for these functions are displayed using functions
      in dwsupport. 

   }

   Purpose: {

      Provide (A)Rexx-like string functions (and others) within REBOL

   }

   Comment: {

      Most of these functions are invoked with the same arguments and
      in the same order as the one of the same name within ARexx. 
      
      Use EXHELP (or HELP) command-name to see more information.

      ARexx-like functions and arguments:

         ABBREV    str abbrev /length len

         BRKSTR    str int /pad pchar /rev

         CENTER    str cnt /pad pchar

         COMPARE   str str2 /case /pad pchar

         COMPRESS  str /pad pchar

         COPIES    str cnt

         DELSTR    str start cnt

         DELWORD   str start cnt

         FMTSTR    str [ints] str2|[str2 str3 ...]

         INSSTR    str2 str  pos

         LASTPOS   pattern str /case /before pos

         LASTWORD  str

         LEFT      str cnt /pad pchar

         OVERLAY   str ovstr start /length len /pad pchar

         POS       pattern str /case /after pos

         REPLSTR   str old new /case

         REVERSE   str

         RIGHT     str cnt /pad pchar

         SCANSTR   str delim /first /n wordnum /last /cnt

         SPACE     str cnt /pad pchar

         STRIP     str /head /tail /pad pchar

         SUBSTR    str start cnt /pad pchar

         SUBWORD   str start cnt

         VERIFY    str str2 /case /match

         WRAPTXT   str cnt  /pad pchar

         View help for the differences between SAFETRANS/TRANSLATE

         SAFETRANS str newchars oldchars /case
         TRANSLATE str newchars oldchars /case

         WORD       str nth

         WORDLENGTH str nth

         WORDS      str

         XRANGE     start end

      Conversion functions:

         The C2* functions expect the file %scripts/d01to255.dat to exist. 
         This file contains all the ascii characters from decimal 1 through 255.
         Change the assignment for word d01to255 if necessary. 


         These two functions can be used to convert between bases. They support
         base 2 through base 36. The base argument must be of type number!

         For the C2* and *2C functions, str must be a single character with 
         a type of string!

         BASE2D   str  base

         D2BASE   int  base

         B2C      str

         B2D      str

         B2X      str

         C2B      str

         C2D      str

         C2X      str

         D2B      int

         D2C      int

         D2X      int

         X2B      str

         X2C      str

         X2D      str


      Additional functions:

         BLOCKWORDS string

            Returns a block of words constructed from the string
            Only spaces are considered as word delimiterers
            translate may be used to convert other characters to spaces

         HELP name

            OLD-HELP name

            This pair changes the original HELP command to OLD-HELP.
            It displays the external help file if able to and calls 
            the orginal help command if the file is not found.

            Delete functions OLD-HELP and HELP if you don't wish 
            to use this method.

         EXHELP name

            Displays helpfile %help/default-language/*.help

         INHELP name

            Displays any inlined information for functions.
            (This method is no longer used by this package)

         LOWER str

            An alias for the REBOL LOWERCASE function

         UPPER str

            An alias for the REBOL UPPERCASE function

   }

   Uninstall: [
   
      "library" "strings/d01to255.dat"
      "library" "strings/dwstrings.rlib"
      "help"    "strings/dwstrings"
      "user"    {loadlib "strings/dwstrings"}

   ]

]

alias 'lowercase "lower"
alias 'uppercase "upper"

abbrev: func [str abbr /length len] [

   if not length [len: 1]

   if all [((length? str) >= len) ((length? abbr) >= len)
           ((pos abbr str) = 1)] [return true] else [return false]
]

blockwords: func [str] [

   use [wb cpos cidx] [
      str: trim/auto str
      wb: make block! 0
      while [found? cpos: find str " "] [
         cidx: index? cpos
         insert/part tail wb str cidx
         str: trim/auto remove/part str cidx
      ]
      insert tail wb str
      return head wb
   ]
]

brkstr: func [str int /pad pchar /rev] [
   if not pad [pchar: " "]
   if rev [
      int: -1 * int
      str: tail str
   ]
   while [true] [
      str: skip str int
      if (tail? str) and (not rev) [break]
      if (head? str) and rev [break]
      str: insert str pchar
      if rev [str: past str]
   ]
   return head str
]


;-- This will be MUCH cleaner if a case-sensitive equals? becomes available.
;-- until then, it is necessary to use find/case on formed characters.

compare: func [str str2 /case /pad pchar] [

   if not all [(string? str) (string? str2)] [
      print ["Arguments must be strings"]
      exit
   ]

   if not pad [pchar: " "]

   use [found ctr] [
      if (length? str) < (length? str2) [
         str: left/pad str length? str2 pchar
      ]
      else [
         if (length? str2) < (length? str) [
            str2: left/pad str2 length? str pchar
         ]
      ]
      ctr: make integer! 0
      found: false
      foreach char str [
         ctr: ctr + 1 
         if case [
            found: found? find/case form pick str2 ctr form char 
            if not found [return ctr]
         ]
         else [
            if char <> (pick str2 ctr) [return ctr]
         ]
      ]
      return 0
   ]
]

compress: func [str /pad pchar] [

   use [newstr cpos] [
      newstr: make string! str
      if not pad [pchar: " "]
      foreach char pchar [
         while [found? cpos: find newstr char] [
            remove/part skip newstr ((index? cpos) - 1) 1
         ]
      ]
      return head newstr
   ]
]

center: func [str cnt /pad pchar] [

   use [newstr lcnt] [
      newstr: make string! cnt
      if not pad [pchar: " "]
      lcnt: make integer! (cnt - (length? str)) / 2
      insert/dup head newstr pchar lcnt
      insert tail newstr str
      insert/dup tail newstr pchar (cnt - (length? str) - lcnt)
      return head newstr
   ]

]

copies: func [str cnt] [

   use [newstr dups] [

     newstr: make string! str
     if cnt < 16 [return head insert/dup newstr str (cnt - 1)]
     dups: make integer! (log-2 cnt)
     loop dups [insert tail newstr head newstr]
     return head insert newstr 
            copy/part newstr ((cnt * (length? str)) - (length? newstr)) 
   ]
]

delstr: func [str start cnt] [

   use [newstr] [
      newstr: make string! str
      cnt: minimum ((length? str) - start + 1)  cnt
      return head remove/part skip newstr (start - 1) cnt
   ]
]

delword: func [str start cnt] [

   use [wb] [
      wb: blockwords str
      cnt: minimum ((length? wb) - start + 1)  cnt
      wb: head remove/part skip wb (start - 1) cnt
      if (length? wb) = 0 [wb: ""]
      return form wb
   ]
]


fmtstr: func [str ints chars] [
   if any [(not string? str) (not block? ints) 
           ((not block? chars) and (not string? chars)) 
           ((length? chars) <> (length? ints))
          ] [print "See help for fmtstr for argument requirements" exit]

   repeat num (length? ints) [
      str: insert skip str pick ints num pick chars num
   ]
   return head str
]


insstr: func [str2 str pos] [

   use [newstr] [
      newstr: make string! str
      return head insert skip newstr pos str2
   ]
]

lastpos: func [pattern str /case /before pos] [

   use [newstr cpos oldpos] [

      newstr: make string! str
      oldpos: make integer! 0

      if before [
         newstr: head clear skip newstr (pos - 1)
      ]
      if case [
         while [found? cpos: find/case skip newstr oldpos pattern] [
            oldpos: index? cpos
         ]
      ]
      else [
         while [found? cpos: find skip newstr oldpos pattern] [
            oldpos: index? cpos
         ]
      ]
      return oldpos
   ]
]

lastword: func [str] [return word str words str]

left: func [str cnt /pad pchar] [

   use [newstr] [
      newstr: copy/part head str cnt
      if not pad [pchar: " "]
      if ((length? newstr) < cnt) [
         insert/dup tail newstr pchar (cnt - (length? newstr))
      ]  
      return head clear skip newstr cnt
   ]
]

overlay: func [str ovstr start /length len /pad pchar] [

   use [newstr] [
      if not pad [pchar: " "]
      if not length [len: length? ovstr]

      newstr: left/pad str (start - 1) pchar

      insert tail newstr left/pad ovstr len pchar

      if ((length? str) - start - len - 1) > 0 [
         insert tail newstr skip str (start + len - 1)
      ]

      return head newstr

   ]
]

pos: func [pattern str /after pos /case] [

   use [cpos] [
      if after [str: skip str pos]
      if case [
         if found? cpos: find/case str pattern [
            return index? cpos
         ]
         else [return 0]
      ]
      else [
         if found? cpos: find str pattern [
            return index? cpos
         ]
         else [return 0]
      ]
   ]
]

replstr: func [str old new /case] [

   use [newstr oldlen newlen start cpos] [

      newstr: make string! str
      oldlen: length? old
      newlen: length? new
      start:  0

      if case [
         while [found? cpos: find/case skip newstr start old] [
             start: (index? cpos) + newlen - 1
             newstr: head remove/part skip newstr ((index? cpos) - 1) oldlen
             newstr: head insert skip newstr ((index? cpos) - 1) new
         ]
      ]
      else [
         while [found? cpos: find skip newstr start old] [
             start: (index? cpos) + newlen - 1
             newstr: head remove/part skip newstr ((index? cpos) - 1) oldlen
             newstr: head insert skip newstr ((index? cpos) - 1) new
         ]
      ]
      return newstr
   ]
]

reverse: func [str] [

   use [newstr] [
      newstr: make string! length? str
      foreach char str [insert newstr char]
      return head newstr
   ]
]


right: func [str cnt /pad pchar] [

   use [newstr] [
      newstr: copy skip str (length? str) - cnt 
      if pad and ((length? newstr) < cnt) [
         insert/dup head newstr pchar (cnt - (length? newstr))
      ]  
      return head newstr
   ]
]

scanstr: func [str delim /first /n wordnum /last /cnt /nostart /doend] [

   use [oldpos cpos newpos slen dlen blk foundit] [

      if not n [wordnum: 0]

      blk: make block! 0
      foundit: false
      slen: length? str
      dlen: (length? form delim) - 1
      oldpos: make integer! 0
      newpos: make integer! 0

      while [true] [
         if found? cpos: find skip str oldpos delim [
            newpos: index? cpos
            if (newpos <> 1) or (not nostart) [
               insert tail blk copy/part skip str oldpos (newpos - oldpos - 1)
            ]
            foundit: true
            oldpos: newpos + dlen 
            if oldpos = slen [
               oldpos: 0
               if doend [
                  insert tail blk ""
               ]
               break
            ]
         ]
         else [break]
      ]

      if foundit [

         if (oldpos > 0) [
            insert tail blk copy/part skip str oldpos (slen - oldpos)
         ]

         if first [wordnum: 1]
         if last [wordnum: length? blk]
         if cnt [return length? blk]
         if wordnum > 0 [return pick blk wordnum]
         return blk
      ]

      if cnt [return 0]
      return none!
   ]
]

space: func [str cnt /pad pchar] [

   use [wb newstr] [
      if not pad [pchar: " "]
      newstr: make string! (length? str) + (cnt * (length? pchar))
      wb: blockwords str 
      for n 1 length? wb 1 [
         newstr: insert tail newstr pick wb n
         if n < (length? wb) [
            insert/dup tail newstr pchar cnt
         ]
      ]
      return head newstr
   ]

]

strip: func [str /head /tail /pad pchar] [

   use [newstr len] [
      newstr: make string! str
      if not pad [pchar: " "]
      if not tail [
         len: (verify newstr pchar) - 1 
         if len > 0 [newstr: delstr newstr 1 len]
      ]
      if not head [
         newstr: reverse newstr
         len: (verify newstr pchar) - 1
         if len > 0 [newstr: delstr newstr 1 len]
         newstr: reverse newstr
      ]
      return newstr
   ]
]


substr: func [str start cnt /pad pchar] [

   use [newstr] [
      newstr: head copy/part skip str (start - 1) cnt
      if (length? newstr) < cnt [
         if not pad [pchar: " "]
         newstr: left/pad newstr cnt pchar
      ]
      return newstr
   ]
]


subword: func [str start cnt] [

   use [wb] [
      wb: remove/part blockwords str start - 1
      wb: head clear skip wb cnt  
      if ((length? wb) = 0) [wb: ""]
      return form wb
   ]
]


translate: func [str newchars oldchars /case] [

   if not all [ (string? str) (string? newchars) (string? oldchars) ] [
      print ["Arguments must be strings"]
      exit
   ]
   if ((length? oldchars) <> (length? newchars)) [
      print "Must be an equal number of OLD and NEW characters"
      exit
   ]

   use [newstr] [
      newstr: make string! str
      if case [
         foreach ochar oldchars [
            while [find/case newstr ochar] [
               change/part find/case newstr ochar newchars 1
            ]
            newchars: next newchars
         ]
      ]
      else [
         foreach ochar oldchars [
            while [find newstr ochar] [
               change/part find newstr ochar newchars 1
            ]
            newchars: next newchars
         ]
      ]
      return newstr
   ]
]

safetrans: func [str newchars oldchars /case] [

   if not all [ (string? str) (string? newchars) (string? oldchars) ] [
      print ["Arguments must be strings"]
      exit
   ]
   if ((length? oldchars) <> (length? newchars)) [
      print "Must be an equal number of OLD and NEW characters"
      exit
   ]

   use [newstr found] [
      newstr: make string! length? str
      foreach char str [
         found: false
         newchars: head newchars
         foreach ochar oldchars [
            if case [
               if found? find/case form char form ochar [
                  insert/part tail newstr newchars 1
                  found: true
               ] 
             ]
             else [
               if found? find form char form ochar [
                  insert/part tail newstr newchars 1
                  found: true
               ] 
             ]
             if found [break]
             newchars: next newchars
         ]
         if not found [insert tail newstr char]
      ]
      return newstr
   ]
]


verify: func [str str2 /case /match] [

   if not all [(string? str) (string? str2)] [
      print ["Arguments must be strings"]
      exit
   ]

   use [found ctr] [
      ctr: make integer! 0
      found: false
      foreach char str [
         ctr: ctr + 1 
         if case [
            found: found? find/case str2 form char 
            if found and match [return ctr]
            if (not found) and (not match) [return ctr]
         ]
         else [
            found: found? find str2 form char 
            if found and match [return ctr]
            if (not found) and (not match) [return ctr]
         ]
      ]
      return 0
   ]
]


word: func [str nth] [

   use [wb] [
      wb: blockwords str
      return pick wb nth
   ]
]

wordlength: func [str nth] [

   use [wb] [
      wb: blockwords str
      return length? pick wb nth
   ]
]

words: func [str] [

   use [wb] [
      wb: blockwords str
      if ((length? first wb) = 0) [return 0]
      else [return length? wb]
   ]
]
 
wraptxt: func [str cnt /pad pchar] [

   use [newstr curpos prepos oldpos inscnt inslen] [

      if (not pad) [pchar: make char! " "]
      else [pchar: make char! pchar]

      newstr: make string! str
      curpos: make integer! 0
      prepos: make integer! 0
      oldpos: make integer! 0
      inscnt: make integer! 0
      inslen: length? form newline

      foreach char str [
         curpos: curpos + 1
         if char = pchar [
            if (curpos - oldpos) <= cnt [prepos: curpos]
         ]
         if (curpos - oldpos) >= cnt [         
            if oldpos = prepos [prepos: curpos]
            insert skip newstr (prepos + inscnt) newline
            oldpos: curpos
            inscnt: inscnt + inslen
         ]
      ]
      return head newstr
   ]
]

xrange: func [start end] [

   if any [(start < 1) (start > 255) (end < 1) (end > 255)] [
      print "Arguments must be numbers from 1 through 255"
      exit
   ]
   use [newstr len] [
      newstr: make string! 0
      if not (end < start) [
         len: end - start + 1
         newstr: copy/part skip d01to255 (start - 1) len
      ]
      else [
         len: start - end + 1
         newstr: copy/part skip d01to255 (end - 1) len
         newstr: reverse newstr
      ]
      return head newstr
   ]
]

base2d: func [str base] [

   if (base < 2) or (base > 36) [
      print "Base value must be 2 through 36"
      exit
   ]
   use [bvals ctr char] [
      bvals: "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ"
      ctr: make integer! 0
      str: reverse str
      for num 1 (length? str) 1 [
         ind: -1
         char: form pick str num
         if (found? find bvals char) [
            ind: (index? find bvals char) - 1
         ]
         if (ind = -1) or (ind > (base - 1)) [
            print [pick str num "is not a valid character for base " base]
            exit
         ]
         ctr: ctr + (ind * (power base (num - 1)))
         
      ]
      return ctr
   ]
]

d2base: func [int base] [

   if (base < 2) or (base > 36) [
      print "Base value must be 2 through 36"
      exit
   ]
   use [bvals bstr] [
      bvals: "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ"
      bstr: make string! 0
      while [int > (base - 1)] [
         insert head bstr pick bvals ((int // base) + 1)
         int: int - (int // base)
         if int > 0 [
            int: int / base
         ]
      ]
      insert head bstr pick bvals (int + 1)
      return head bstr
   ]
]


b2c: func [str] [

    return form d2c make integer! base2d str 2
]


b2d: func [str] [

    return make integer! base2d str 2
]


b2x: func [str] [

    return d2base make integer! base2d str 2 16
]


c2b: func [str] [

    return right/pad d2base c2d str 2 8 "0"

]


c2d: func [str] [

   if (length? str) > 1 [
      print "C2D argument must be a single character of type string!"
      exit
   ]

   return (index? find/case d01to255 str)
]

c2x: func [str] [

    return right/pad d2x c2d str 2 "0"

]

d2b: func [int] [

   use [a rem] [
      a: d2base int 2
      rem: (length? a) // 4
      if rem = 0 [return a]
      else [
         return right/pad a ((length? a) + 4 - rem) "0"
      ]
   ]
]


d2c: func [int] [

   return form pick d01to255 int

]


d2x: func [int] [

   return d2base int 16

]

x2b: func [str] [

   use [a rem] [
      a: d2base make integer! base2d str 16 2
      rem: (length? a) // 4
      if rem = 0 [return a]
      else [
         return right/pad a ((length? a) + 4 - rem) "0"
      ]
   ]
]


x2c: func [str] [

   return form pick d01to255 make integer! base2d str 16

]


x2d: func [str] [

   return make integer! base2d str 16
]

d01to255: read make file! compress form [lib-path "strings/d01to255.dat"]

;**************************************************************************;
;*           Alternate method, but VERY slow on my 040/25 Amiga           *;
;**************************************************************************;

;d01to255func: func [] [
;
;   d01to255: make string! 255
;   for n 1 255 1 [insert tail d01to255 form make char! n]
;   
;]
;
;d01to255func
;unset 'd01to255func


""}

]

;**************************************************************************;
;*                   Files to be generated "on the fly"                   *;
;*   Third entry is a block of code that has a "head do" performed on it  *;
;*                 Result is written as in file-to-install                *;
;**************************************************************************;

files-to-generate: [

   "library"
   "strings/d01to255.dat" 
   [ascstr: make string! 255 
      for n 1 255 1 [insert tail ascstr make char! n]
   ]
]

;**************************************************************************;
;*    Function names in this package. Used for help and example files     *;
;**************************************************************************;
;*           Include file is created by build-fnames.r script             *;
;**************************************************************************;

func-names: 
["abbrev" "b2c" "b2d" "b2x" "base2d" "blockwords" "brkstr" "c2b" "c2d" "c2x" "center" "compare" "compress" "copies" "D01to255" "d2b" "d2base" "d2c" "d2x" "delstr" "delword" "dwstrings" "fmtstr" "insstr" "lastpos" "lastword" "left" "lower" "overlay" "pos" "replstr" "reverse" "right" "safetrans" "scanstr" "space" "strip" "substr" "subword" "translate" "upper" "verify" "word" "wordlength" "words" "wraptxt" "x2b" "x2c" "x2d" "xrange"]


;**************************************************************************;
;*                         Help and Example files                         *;
;*                    File created using build-help.r                     *;
;**************************************************************************;

; START OF HELP INCLUDES


abbrev.help: {
Usage: ABBREV str abbrev /length len

    Returns TRUE if abbrev is an abbreviation of str, FALSE otherwise.

    Use /length to set a minimum allowable length for abbrev to len.
    Default length is 1.
}
b2c.help: {
Usage: B2C str

    Returns the ascii character corresponding to binary value of str

    See also: B2D B2X C2B C2D C2X D2B D2C D2X X2B X2C X2D
}
b2d.help: {
Usage: B2D str

    Returns the decimal equivalent of the binary str

    See also: B2C B2X C2B C2D C2X D2B D2C D2X X2B X2C X2D
}
b2x.help: {
Usage: B2X str

    Returns the hex value for the binary str

    See also: B2C B2D C2B C2D C2X D2B D2C D2X X2B X2C X2D
}
base2d.help: {
Usage: BASE2D str base

    Returns the decimal value of a string in base representation.
    Valid values for base are 2 through 36.
}
blockwords.help: {
Usage: BLOCKWORDS str

    Returns a block of words constructed from the string
    Only spaces are considered as word delimiterers
    translate may be used to convert other characters to spaces
}
brkstr.help: {
Usage: BRKSTR str int /pad pchar /rev

    Is used to insert a character after every int characters in str.
    The default insert character is a space.

    /pad refinement uses pchar instead of space.
    /rev refinement processes the string from right to left.

    See also: FMTSTR
}
c2b.help: {
Usage: C2B str

    Returns the binary value for a character as a string. 
    Character NULL is not handled.

    C2B requires the input to be a single character of type string!

    See also: B2C B2D B2X C2D C2X D2B D2C D2X X2B X2C X2D
}
c2d.help: {
Usage: C2D str

    Returns the decimal value for a character. Character NULL is not handled.

    C2D requires the input to be a single character of type string!

    See also: B2C B2D B2X C2B C2X D2B D2C D2X X2B X2C X2D
}
c2x.help: {
Usage: C2X str

    Returns the hex value for a character as a string. 
    Character NULL is not handled.

    C2X requires the input to be a single character of type string!

    See also: B2C B2D B2X C2B C2D D2B D2C D2X X2B X2C X2D
}
center.help: {
Usage: CENTER str cnt /pad pchar

    Returns a string centered in a space padded string of length cnt
    if /pad is specified, then pchar is used as the padding string
}
compare.help: {
Usage: COMPARE str str2 /case /pad pchar

    Returns the position of the first character in str that is NOT the same
    as the character at that position in str2. 

    Returns 0 if all are the same. 

    Comparing is case sensitive if /case is specified.

    Shorter string is padded with spaces or with pchar if /pad is specified.
}
compress.help: {
Usage: COMPRESS str /pad pchar

    Returns a string with all spaces removed. If the /pad refinement
    is used, all occurrences of each pchar are removed instead of spaces.
}
copies.help: {
Usage: COPIES str cnt

    Returns a string of n copies of str
}
D01to255.help: {
D01to255.dat is a data file with all ASCII values from decimal 01 to 255.
It is used by the XRANGE, C2D, D2C, and X2C functions.
}
d2b.help: {
Usage: D2B int

    Returns a binary string equivalent to decimal int.

    See also: B2C B2D B2X C2B C2D C2X D2C D2X X2B X2C X2D
}
d2base.help: {
Usage: D2BASE int base

    Returns a string with base value of int.
}
d2c.help: {
Usage: D2C int

    Returns the ascii character corresponding to DECIMAL value of int.
    Returns none for out of range values of int.

    See also: B2C B2D B2X C2B C2D C2X D2B D2X X2B X2C X2D
}
d2x.help: {
Usage: D2X int

    Returns the base16 (hexadecimal) value of int.

    See also: B2C B2D B2X C2B C2D C2X D2B D2C X2B X2C X2D
}
delstr.help: {
Usage: DELSTR str start cnt

    Returns a string with start characters for cnt removed

    See also: INSSTR DELWORD
}
delword.help: {
Usage: DELWORD str start cnt

    Returns a string with words start for cnt removed

    See also: DELSTR INSSTR 
}
dwstrings.help: {
Included STRING functions:

ABBREV     tests if one string is an abbreviation of another
BLOCKWORDS makes a block of words from a string
CENTER     centers text within a specified length string
COMPARE    compares two strings
COMPRESS   removes all occurrences of spaces or other characters
COPIES     makes nn copies of a string
DELSTR     removes a portion of a string
DELWORD    removes specified words from a string
INSSTR     inserts one string within another
LASTPOS    finds the position of the last occurrence of one string in another
LASTWORD   returns the last word from a string
LEFT       returns the leftmost nn characters of a string
LOWER      converts all characters in a string to lower case
OVERLAY    overlays one string on another
POS        finds the first position of one string within another
REPLSTR    replaces all occurrences of one string with another
REVERSE    returns a string reversed right-to-left
RIGHT      returns the rightmost nn characters of a string
SAFETRANS  translates all occurrences of characters within a string
SCANSTR    parses string into words using specified delimiter. 
SPACE      inserts nn spaces or other characters between words in a string
STRIP      strips leading and/or trailing spaces or other characters from a string
SUBSTR     returns a sub-string of a string
SUBWORD    returns the specified space delimited words from a string
TRANSLATE  translates all occurrences of characters within a string
UPPER      converts all characters in a string to upper case
VERIFY     finds first character that isn't in a set or first that is in set
WORD       returns the specified word of a string
WORDLENGTH returns the length of the specified word of a string
WORDS      returns the number of words in a string
XRANGE     returns a string with all characters in the specified range

Included CONVERSION functions:

BASE2D   convert base2 thru base36 string to decimal value
D2BASE   convert decimal value to base2 thru base36 string

B2C binary to character      C2B character to binary
B2D binary to decimal        C2D character to decimal
B2X binary to hex            C2X character to hex


D2B decimal to binary        X2B hex to binary
D2C decimal to character     X2C hex to character
D2X decimal to hex           X2D hex to decimal
}
fmtstr.help: {
Usage: FMTSTR str [ints] [str1 str2 ...]
       FMTSTR str [ints] str2   

    Returns a string with characters or strings inserted after the number
    of characters in the input string specified by the block of ints.

    If the third argument is a block of strings, then each string is inserted
    after the corresponding point from ints.

    If the third argument is a string, then a single character of string is
    inserted after the corresponding point from ints.

    See also: BRKSTR
}
insstr.help: {
Usage: INSSTR str2 str pos

    Returns a string with str2 inserted after pos

    See also: DELSTR DELWORD
}
lastpos.help: {
Usage: LASTPOS pattern str /case /before pos

    Returns the position of the last occurrence of pattern
    BEFORE pos if /before is specified else before the end of str

    Returns 0 if not found

    /CASE makes the pattern matching case sensitive.

    See also: POS
}
lastword.help: {
Usage: LASTWORD str

    Returns the LAST space delimited word of a string

    See also: WORD SUBWORD 
}
left.help: {
Usage: LEFT str cnt /pad pchar

    Returns the leftmost n characters of a string
    padded to length cnt with spaces or pchar if /pad 
    is specified

    See also: CENTER RIGHT SUBSTR
}
lower.help: {
Usage: LOWER str 

    An alias for REBOL's built-in LOWERCASE function
    Returns the string converted to all lower-case values

    See also: UPPER TRANSLATE SAFETRANS
}
overlay.help: {
Usage: OVERLAY str ovstr start /length len /pad pchar

    Returns a string with the orginal overlayed by ovstr at position start.
    If start is greater than the length of the original string, the string 
    is padded with spaces before being overlayed. If /pad is specified, then
    pchar is used for the padding. If /length is specified, then the ovstr
    is truncated or padded, using pchar or spaces, to len prior to the overlay.

    See also: BRKSTR FMTSTR
}
pos.help: {
Usage: POS pattern str /case /after pos

    Returns the position of the first occurrence of pattern
    AFTER pos if /after is specified else after beginning of str

    Returns 0 if not found

    /CASE makes the pattern matching case sensitive.

    See also: LASTPOS
}
replstr.help: {
Usage: RELPSTR str old new /case

    Replaces all occurrences of old with new in str

    /case makes substitution case sensitive

    See also: DELSTR INSSTR
}
reverse.help: {
Usage: REVERSE str

    Returns a string in reversed order
}
right.help: {
Usage: RIGHT str cnt /pad pchar

    Returns the rightmost n characters of a string
    padded with pchar if /pad is specified

    See also: CENTER LEFT SUBSTR
}
safetrans.help: {
Usage: SAFETRANS str newchars oldchars /case

    Returns a string with all occurrences of oldchars changed to newchars
    Character replacement processes each character in string only once.
    Use this one to prevent changing a character multiple times.

    If /case is specified then matching will be case sensitive

    See also: TRANSLATE UPPER LOWER
}
scanstr.help: {
Usage: SCANSTR str delim /first /n wordnum /last /cnt /nostart /doend

      Parses string into words using specified delimiter.

      scanstr       str delim -- return a block with words
      scanstr/cnt   str delim -- return the count of words or 0
      scanstr/first str delim -- return the first word
      scanstr/last  str delim -- return the last word
      scanstr/n     str delim 8 -- return the 8th word

      The default behavior of scanstr is to include everything UPTO a
      delimiter, thus: 

         scanstr/cnt "/dir1/dir2/" "/" 

      returns a count of 3 with the first string being of zero length 
      and NOT counting a null after the last delimiter. 

      You can change this behavior by:

      scanstr/nostart str delim -- DON'T include a null start value

      scanstr/doend   str delim -- DO include an ending null value

      All return none! (except /cnt) if no delimiter is found.
}
space.help: {
Usage: SPACE str cnt /pad pchar

    Returns a string of words with cnt spaces separating them.
    If /pad is specified, then cnt occurrences of pchar are used
    for spacing. Leading and trailing spaces are removed.

    See also: BRKSTR COMPRESS FMTSTR OVERLAY 
}
strip.help: {
Usage: STRIP str /head /tail /pad pchar

    Returns a string with leading and trailing spaces removed.     
    Use /pad to remove 'pchar' instead. 
    'pchar' should be a single character of type string!

    Specify /head to remove only leading spaces or 'pchar'
    Specify /tail to remove only trailing spaces or 'pchar' 
}
substr.help: {
Usage: SUBSTR str start cnt /pad pchar

    Returns a string from start character for cnt. If the resultant
    string is shorter than cnt, then it is padded with spaces or 
    with pchar if /pad is specified. 

    See also: LEFT RIGHT
}
subword.help: {
Usage: SUBWORD str start cnt

    Returns a string of words from start for cnt

    See also: LASTWORD WORD
}
translate.help: {
Usage: TRANSLATE str newchars oldchars /case

    Returns a string with all occurrences of oldchars changed to newchars
    Character replacement proceeds using the value of the first oldchar 
    for the entire string, then the second oldchar, etc. 

    If /case is specified then matching will be case sensitive

    TRANSLATE is MUCH faster than SAFETRANS, but characters in the string may 
    be changed by overlapping changes of oldchar to newchar.

    See also: SAFETRANS UPPER LOWER
}
upper.help: {
Usage: UPPER str 

    An alias for REBOL's built-in upperCASE function
    Returns the string converted to all upper-case values

    See also: LOWER TRANSLATE SAFETRANS
}
verify.help: {
Usage: VERIFY str str2 /case /match

    Returns the position of the first character in str that is NOT in str2.
    If str is composed of only characters found in str2 then returns 0.


    If /case is specified then verifying will be case sensitive

    If /match is specified then returns the position of the first character
    in str that IS in str2. 
    Returns 0 if str is composed of only characters NOT in str2.
}
word.help: {
Usage: WORD str nth

    Returns the nth space delimited word of a string

    See also: LASTWORD SUBWORD 
}
wordlength.help: {
Usage: WORDLENGTH str nth

    Returns the length of the nth word of the string

    See also: WORDS
}
words.help: {
Usage: WORDS str

    Returns the number of blank delimited words in a string

    See also: SUBWORD WORD WORDLENGTH 
}
wraptxt.help: {
Usage: WRAPTXT str cnt /pad pchar

    Returns a string broken into approximately cnt length pieces, with newlines
    inserted between them. Breaks the string after spaces, if possible (or after
    pchar if /pad is specified).

    See also: BRKSTR FMTSTR SCANSTR
}
x2b.help: {
Usage: X2B str

    Returns a binary string for the hexadecimal string

    See also: B2C B2D B2X C2B C2D C2X D2B D2C D2X X2C X2D
}
x2c.help: {
Usage: X2C str

    Returns the ascii character corresponding to the hexadecimal string

    See also: B2C B2D B2X C2B C2D C2X D2B D2C D2X X2B X2D
}
x2d.help: {
Usage: X2D str

    Returns the decimal value of hexadecimal string

    See also: B2C B2D B2X C2B C2D C2X D2B D2C D2X X2B X2C
}
xrange.help: {
Usage: XRANGE start end

    Returns a string consisting of all the ascii characters 
    from decimal 'start' through decimal 'end'

    Valid values for start and end are 1 through 255

    If 'end' is less than 'start' the string will be in the
    reverse of normal order.

    See also: B2D C2D X2D
}
abbrev.example: {[
{print abbrev "north" "nor"}
{print abbrev/length "north" "nor" 4}
{print abbrev "north" "not"}
]
}
b2c.example: {[
{print b2c "01000001"}
{print b2c compress "0110 0001"}
]
}
b2d.example: {[
{print b2d "01000001"}
{print b2d compress "0110 0001"}
]
}
b2x.example: {[
{print b2x "01000001"}
{print b2x compress "0110 0001"}
]
}
base2d.example: {[
{print base2d "1000" 2}
{print base2d "FF" 16}
{print base2d "ZZ" 36}
]
}
blockwords.example: {[

{use [blk] [
   blk: blockwords "this is some text"
   foreach word blk [print [word]]
]}

]
}
brkstr.example: {[
{print [brkstr "11010010" 4]}
{print [brkstr/pad "9999999" 3 ","]}
{print [brkstr/pad/rev "9999999" 3 ","]}
]
}
c2b.example: {[
{print c2b "A"}
{print c2b "a"}
]
}
c2d.example: {[
{print c2d "A"}
{print c2d "a"}
]
}
c2x.example: {[
{print c2x "A"}
{print c2x "a"}
]
}
center.example: {[
{print [center "this is some text" 40]}
{print [center/pad "this is some text" 40 "="]}
]
}
compare.example: {[
{print [compare "this is" "THIS IS"]}
{print [compare/case "this is" "THIS IS"]}
{print [compare "this is" "this iX"]}
{print [compare "this is" "this is   "]}
{print [compare/pad "this is" "this is   " "+"]}
]
}
compress.example: {[
{print compress "this is a test"}
{print compress/pad "this--is--a--test" "-"}
{print compress/pad "-;-this -;-is -;-a -;-test-;-" ";-"}
]
}
copies.example: {[
{print copies "testing..." 3}
{print copies "=+=" 20}
]
}
d2b.example: {[
{print d2b 65}
{print d2b 97}
]
}
d2base.example: {[
{print d2base  8 2}
{print d2base  255 16}
{print d2base  1295 36}
]
}
d2c.example: {[
{print d2c 65}
{print d2c 97}
]
}
d2x.example: {[
{print d2x 65}
{print d2x 97}
]
}
delstr.example: {[
{print [delstr "this is not a test" 1 8]}
{print [delstr "this is not a test" 9 4]}
]
}
delword.example: {[
{print [delword "this is not a test" 1 2]}
{print [delword "this is not a test" 3 1]}
]
}
fmtstr.example: {[
{print [fmtstr "7074678000" [0 3 0 3] "() -"]}
{print [fmtstr "7074678000" [0 3 3]   ["(" ") " "-"]]}
{print [fmtstr "7074678000" [0 3 3]   ["Area code: " "  Phone: " "-"]]}
]
}
insstr.example: {[
{print [insstr "not " "this is a test" 8]}
{print [insstr "this is " "not a test" 0]}
]
}
lastpos.example: {[
{print [lastpos "not " "this is NOT a test"]}
{print [lastpos/case "not" "this is NOT a test"]}
{print [lastpos "is" "this is a test"]}
{print [lastpos/before "is" "this is a test" 5]}
]
}
lastword.example: {[
{print [lastword "this is not a test"]}
]
}
left.example: {[
{print [left "this is a test" 7]}
{print [left/pad "test" 10 "*"]}
]
}
lower.example: {[
{print [lower "THIS IS A TEST"]}
]
}
overlay.example: {[
{print [overlay "this is a test" "WHAT"   1]}
{print [overlay/length "this is a test" "attilla" 3 2]}
{print [overlay/pad/length "this is a test" "WHAT" 1 10 "*"]}
]
}
pos.example: {[
{print [pos "not " "this is NOT a test"]}
{print [pos/case "not" "this is NOT a test"]}
{print [pos "is" "this is a test"]}
{print [pos/after "is" "this is a test" 4]}
]
}
replstr.example: {[
{print [replstr "On this date..." "date" form now]}
{print [replstr/case "On date: DATE " "DATE" form now]}
]
}
reverse.example: {[
{print [reverse "tset a si siht"]}
{print [reverse string!]}
]
}
right.example: {[
{print [right "this is a test" 6]}
{print [right/pad "test" 10 "*"]}
]
}
safetrans.example: {[
{print [safetrans "http://rebol.com" "   " "/:."]} 
{print [safetrans "This iS a test" "xz" "ts"]}
{print [safetrans/case "This iS a test" "XZ" "TS"]}
]
}
scanstr.example: {[
{print [scanstr "E7:/REBOL/Library/strings/" "/"]}
{print [scanstr/cnt "E7:/REBOL/Library/strings/" "/"]}
{print [scanstr/cnt "/REBOL/Library/strings/" "/"]}
{print [scanstr/cnt/nostart "/REBOL/Library/strings/" "/"]}
{print [scanstr/cnt/doend "/REBOL/Library/strings/" "/"]}
{print [scanstr/first "E7:/REBOL/Library/strings/" "/"]}
{print [scanstr/last "E7:/REBOL/Library/strings/" "/"]}
{print [scanstr/n "E7:/REBOL/Library/strings/" "/" 3]}
]
}
space.example: {[
{print [space "this is a test" 3]}
{print [space "   this     is     a      test  " 1]}
{print [space/pad "this is a test" 3 "*"]}
]
}
strip.example: {[
{print [strip "      this is a test    "]}
{print [strip/pad "****this is a test ****" "*"]}
{print [strip/head/pad "****this is a test****" "*"]}
]
}
substr.example: {[
{print [substr "this is a test" 6 4]}
{print [substr/pad "test" 1 10 "*"]}
]
}
subword.example: {[
{print [subword "this is a test" 1 3]}
{print [subword "this is a test" 2 2]}
]
}
translate.example: {[
{print [translate "http://rebol.com" "   " "/:."]} 
{print [translate "This iS a test" "xz" "ts"]}
{print [translate/case "This iS a test" "XZ" "TS"]}
]
}
upper.example: {[
{print [upper "this is a test"]}
]
}
verify.example: {[
{print [verify "98765" "0123456789"]}
{print [verify "98X765Z" "0123456789"]}
{print [verify/match "ABC98X765Z" "0123456789"]}
{print [verify/case/match "a985aBc" "AB"]}
]
}
word.example: {[
{print [word "this is a test" 2]}
{print [word "this is a test" 4]}
]
}
wordlength.example: {[
{print [wordlength "this is a test" 2]}
{print [wordlength "this is a test" 8]}
]
}
words.example: {[
{print [words "this is a test"]}
{print [words ""]}
]
}
wraptxt.example: {[
{print [wraptxt "this is a test" 5]}
{print [wraptxt/pad "this is |a test" 10 "|"]}
]
}
x2b.example: {[
{print x2b "FA"}
{print x2b "44"}
]
}
x2c.example: {[
{print x2d "41"}
{print x2d "61"}
]
}
x2d.example: {[
{print x2d "ff"}
{print x2d "ffff"}
]
}
xrange.example: {[
{print xrange 65 90}
{print xrange 90 65}
{print xrange c2d "A" c2d "Z"}
]
}


; END OF HELP INCLUDES

;**************************************************************************;
;*                     FUNCTIONS USED WITHIN INSTALLER                    *;
;**************************************************************************;

;**************************************************************************;
;*         Create absolute paths to %library and %help directories        *;
;**************************************************************************;

ins-loadpaths: func [] [

   use [substr lastpos bootfile] [

      substr: func [str start cnt] [
         use [newstr] [
            newstr: head copy/part skip str (start - 1) cnt
            return newstr
         ]
      ]

      lastpos: func [pattern str] [
         use [found newstr] [
            found: false
            newstr: make string! str
            while [found? find newstr pattern] [
               newstr: next find newstr pattern 
               found: true
            ]
            if found [return (index? newstr) - 1]
            else [return 0]
         ]
      ]

      ;*******************************************;
      ;* Handle v1.0.3.x rebol/bootfile problems *;
      ;*******************************************;

      if block? rebol/bootfile [
         bootfile: first rebol/bootfile
      ]
      else [
         bootfile: rebol/bootfile
      ]

      ins-rebol-path: make string! substr bootfile 1 ((lastpos "/" bootfile) - 1)

      if (exists? ins-rebol-path) [
         ins-rebol-path: head insert tail ins-rebol-path "/"
         ins-help-path:  make string! ins-rebol-path
         ins-help-path:  head insert tail ins-help-path "help/"
         ins-lib-path:   make string! ins-rebol-path
         ins-lib-path:   head insert tail ins-lib-path "library/"
         return true
      ] 
      else [return false]
   ]
]

;**************************************************************************;
;*                Get response from user. "Q" quits script                *;
;**************************************************************************;

ins-getans: func [] [

   ans: input
   if (length? ans) > 0 [ans: form first ans] else [ans: "Y"]
   if ans = "q" [halt] 
   return ans

]

;**************************************************************************;
;*                      Check that directories exist                      *;
;**************************************************************************;
ins-checkdirs: func [] [

   foreach [dtype dpath] required-dirs [
      if dtype = "library" [fullpath: make file! ins-lib-path]
      else [
         if dtype = "help" [fullpath: make file! ins-help-path]
      ]
      fullpath: head insert tail fullpath dpath
      while [not exists? fullpath] [
         print ["You need to create directory: " fullpath]
         prin "(Y/q): "
         ins-getans
      ]
   ]
]
;**************************************************************************;
;*                         INSTALLER STARTS HERE                          *;
;**************************************************************************;

;**************************************************************************;
;*    Let the user know what is going on. Give 'em a chance to stop it    *;
;*          Save each script and data file. Save the help files.          *;
;**************************************************************************;


if ((length? intromsg) > 0) [
   print intromsg
   prin "(Y/q): "
   ins-getans
]

if ((length? changesmsg) > 0) [
   print changesmsg
   prin "(Y/q): "
   ins-getans
]

if not ins-loadpaths [

   print "Unable to determine required path information!"
   print {

   Send bug report to:

      Dick Whiting <dwhiting@europa.com>
   }          
   print "Quitting install script..."
   halt
]

if ((length? byhandmsg) > 0) [
   print byhandmsg
   prin "(Y/q): "
   ins-getans
]

if ((length? required-dirs) > 0) [ins-checkdirs]

print ""
print "Starting Install.."
print ""
print {Enter "Q" at any prompt to Quit, "N" to skip to next step}
print ""


;**************************************************************************;
;*     Save each file, included as a string, to the specified filename    *;
;**************************************************************************;

foreach [dtype fpath file] files-to-install [

   if dtype = "library" [fullpath: make file! ins-lib-path]
   else [
      if dtype = "help" [fullpath: make file! ins-help-path]
   ]

   fullpath: head insert tail fullpath fpath

   print ""
   print ["Save" fullpath "?"]
   prin "(Y/n/q): "
   ins-getans

   if ans = "Y" [ 
      if (exists? fullpath) [
         print ""
         print [fullpath "already exists.. Overwrite it?"]
         prin "(Y/n/q): "
         ins-getans
      ]
      if ans = "Y" [
         write fullpath file
      ]
   ]
]

;**************************************************************************;
;*     If any files generated, generate and save them                     *;
;**************************************************************************;

foreach [dtype fpath filecode] files-to-generate [

   if dtype = "library" [fullpath: make file! ins-lib-path]
   else [
      if dtype = "help" [fullpath: make file! ins-help-path]
   ]

   fullpath: head insert tail fullpath fpath

   print ""
   print ["Save" fullpath "?"]
   prin "(Y/n/q): "
   ins-getans

   if ans = "Y" [ 
      if (exists? fullpath) [
         print ""
         print [fullpath "already exists.. Overwrite it?"]
         prin "(Y/n/q): "
         ins-getans
      ]
      if ans = "Y" [
         write fullpath head do filecode
      ]
   ]
]


;**************************************************************************;
;*     Save a help-file for each function name in the func-names list     *;
;*                    Save an examples file, if present                   *;
;**************************************************************************;

print ""
print "Save help files? NO checking for already existing is done"
prin "(Y/n/q): "
ins-getans

if ans <> "n" [
   foreach [name] func-names [
   
      helpvar: make string! ".help"
      helpvar: head insert head helpvar name
   
      helpfile: make string! ins-help-path
      helpfile: insert tail helpfile script-help
      helpfile: insert tail helpfile "/"
      helpfile: insert tail helpfile name
      helpfile: head insert tail helpfile ".help"

      htest: make string! "value? '"
      htest: head insert tail htest helpvar

      if (do htest) [
         write make file! helpfile mold do helpvar
      ]

      examvar: make string! ".example"
      examvar: head insert head examvar name
   
      examfile: make string! ins-help-path
      examfile: insert tail examfile script-help
      examfile: insert tail examfile "/"
      examfile: insert tail examfile name
      examfile: head insert tail examfile ".example"
      
      etest: make string! "value? '"
      etest: head insert tail etest examvar

      if (do etest) [
         do form ["write make file!" examfile examvar]
      ]
   
   ]
]

;**************************************************************************;
;*                             Final messages                             *;
;**************************************************************************;

if ((length? finalmsg) > 0) [
   print Finalmsg
]

print ""
print "You can delete this install script now"
print ""
print "-- bug reports, comments, etc. to Dick Whiting <dwhiting@europa.com>"
print ""
print "done.."

print "Press any key to Quit"
ins-getans

;**************************************************************************;
;*           Quit so that variables, functions don't carry over.          *;
;**************************************************************************;

quit
