REBOL [
   Title:   "Aminet Tools"
   Author:  "Dick Whiting"
   Email:   dwhiting@europa.com
   Home:    http://www.europa.com/~dwhiting
   Date:    18-Jan-1999
   File:    %scripts/amitools.r
   Version: 1.0.0

   Uses: {

      Functions from dwstrings.rlib and dwsupport.rlib
   }

   Purpose: {

      Provide a set of tools for locating, reading, and downloading 
      files from Aminet

   }

   Functions: {

      amichart  -- get current Aminet chart
      amifind   -- do an Aminet search for a string
      amiget    -- download a file from Aminet 
      amiread   -- get a readme from Aminet
      amirecent -- get recents (various options) from Aminet
      amitools  -- function allows for getting general amitools help
   }

   Tailoring: {

      download-dir -- directory you want files downloaded to..SET THIS
      aminet-http  -- http address for searching
      aminet-ftp   -- ftp  address for getting files from
      aminet-recent -- SET THIS to the number obtained on the Aminet recent
                       page of your aminet-http site. Use the number without
                       quotes or the "?" mark. (e.g. aminet-recent: 12345)
   }

   Uninstall: ["scripts" "amitools"]

]

use [download-dir aminet-recent aminet-http aminet-ftp] [

;**************************************************************************;
;*                                  SETUP                                 *;
;**************************************************************************;

   download-dir: "Download:/"  ; your directory for downloads 

   aminet-recent: none!   ; set this to the number from the Aminet recent page 

   aminet-http: "http://www.aminet.org/"
   aminet-ftp:  "ftp://wuarchive.wustl.edu/systems/amiga/aminet/"


amitools: func [] [{
;**************************************************************************;
;*                                Amitools                                *;
;**************************************************************************;
;*                                                                        *;
;*  Functions:                                                            *;
;*                                                                        *;
;*     amichart  -- get current Aminet chart                              *;
;*     amifind   -- do an Aminet search for a string                      *;
;*     amiget    -- download a file from Aminet                           *;
;*     amiread   -- get a readme from Aminet                              *;
;*     amirecent -- get recents (various options) from Aminet             *;
;*                                                                        *;
;**************************************************************************;
}]

amifind: func ['atext] [{

;**************************************************************************;
;*                           Do an Aminet search                          *;
;**************************************************************************;
;*                                                                        *;
;*  Format:                                                               *;
;*                                                                        *;
;*   amifind atext                                                        *;
;*                                                                        *;
;*  Example usages (keywords can be combined by using a '+' between them) *;
;*                                                                        *;
;*   amifind urb.lha              ; look for file urb.lha                 *;
;*   amifind urb+config           ; look for 'urb' AND 'config'           *;
;*                                                                        *;
;**************************************************************************;

}

   use [afound ablk lnum atmp amsg aname ainfo aread foundtext] [

      afound: read make url! compress form [aminet-http "aminetbin/find?" form atext]

      if not found? find afound "files matching" [
         return afound
      ]

      ablk: scanstr afound newline
      lnum: length? ablk
      foundtext: make string! (length? afound)

      for n 1 (length? ablk) 1 [
         if found? find pick ablk n "files matching" [
            amsg: pick ablk n 
  
            insert tail foundtext amsg
            insert tail foundtext newline
            if (word amsg 1) = "0" [return head foundtext]
         ]
         else [
            if found? find pick ablk n "Size Age Description" [
               insert tail foundtext substr pick ablk n 4 ((length? pick ablk n) - 7)
               insert tail foundtext newline
               lnum: n + 1
               break
            ]
         ]
      ]
   
      for num lnum (length? ablk) 1 [
          if (pick ablk num) = "</PRE>" [break]
          atmp: scanstr pick ablk num ">"
          if (pick atmp 1) = "<hr" [atmp: remove/part atmp 1]
          aname: substr pick atmp 2 1 ((pos "<" pick atmp 2) - 1)
          ainfo: substr pick atmp 3 2 ((pos "<" pick atmp 3) - 3)
          aread: substr pick atmp 4 1 ((pos "<" pick atmp 4) - 1)
          aread: replstr aread "&amp;" "&"
          insert tail foundtext form [aname ainfo aread]
          insert tail foundtext newline

      ]
      return head foundtext
   ]
]

amichart: func [] [{

;**************************************************************************;
;*                      Get the current Aminet Chart                      *;
;**************************************************************************;
;*                                                                        *;
;*  Format:  amichart     ; no arguments                                  *;
;*                                                                        *;
;**************************************************************************;
}

      return read make url! compress 
         form [aminet-http "aminet/aminet/CHARTS"]
]


amiget: func ['afile 'adir] [{

;**************************************************************************;
;*                       Download a file from Aminet                      *;
;**************************************************************************;
;*                                                                        *;
;*  Format:  amiget filename  aminet-directory                            *;
;*                                                                        *;
;*  Example: amiget URB.lha util/misc                                     *;
;*                                                                        *;
;**************************************************************************;
}

      write make file! compress form [download-dir afile] 
          read make url! compress form [aminet-ftp adir "/" afile]
]

amiread: func ['afile 'adir] [{

;**************************************************************************;
;*                      Get a readme file from Aminet                     *;
;**************************************************************************;
;*                                                                        *;
;*  Format:  amiread file aminet-directory                                *;
;*                                                                        *;
;*  Example: amiget URB util/misc                                         *;
;*                                                                        *;
;*    NOTE: this works well with my 'more' function 'more amiread ...'    *;
;**************************************************************************;
}

      return read make url! compress 
         form [aminet-http "aminet/dirs/aminet/" adir "/" afile ".readme"]
]

amirecent: func [/date /dir /n rnum] [{

;**************************************************************************;
;*                  Get the Aminet recent uploads listing                 *;
;**************************************************************************;
;*                                                                        *;
;*  Format:  amirecent /date /dir /n rnum                                 *;
;*                                                                        *;
;*   Options are by date, directory, or personal recent (default) if the  *;
;*   variable aminet-recent is set to your number. If it isn't set, then  *;
;*   sorted by directory is the default if /date isn't specified.         *;
;*                                                                        *;
;*  Example: amirecent/date      ; get full recent sorted by date         *;
;*           amirecent/n 12345   ; get personal recent for number 12345   *;
;*                                                                        *;
;*    NOTE: this works well with my 'more' function 'more amirecent...'   *;
;**************************************************************************;
}

   use [aurl afound ablk foundtext lnum atmp amsg aname ainfo aread] [

      if (not n) and (not aminet-recent) [rnum: none!]
      else [
         if not n [rnum: aminet-recent]
      ]

      if all [(not none? rnum) (not date) (not dir)] [
         aurl: make url! compress 
                        form [aminet-http "aminetbin/recent?" form rnum]
      ]
      else [
         if date [
            aurl: make url! compress form [aminet-http "~aminet/recent.html"]
         ]
         else [
            aurl: make url! compress form [aminet-http "~aminet/recentd.html"]
         ]

      ]   

      afound: read aurl
      ablk: scanstr afound newline
      lnum: length? ablk
      foundtext: make string! (length? afound)

      if (not found? find afound "Recent uploads to Aminet") and
         (not found? find afound "New Aminet uploads")  [
         return afound
      ]

      if rnum [
         if found? find afound "No new files since your last call" [
            return "No new files since your last call"
         ]

         for bnum 1 (length? ablk) 1 [
            if (pick ablk bnum) = "<pre>" [
               lnum: bnum + 1
               break
            ]
         ]         
      ]
      else [
         for bnum 1 (length? ablk) 1 [
            if found? find pick ablk bnum "Size Age Description" [
               insert tail foundtext substr 
                        pick ablk bnum 4 ((length? pick ablk bnum) - 7)
               insert tail foundtext newline
               lnum: bnum + 1
               break
            ]
         ]
      ]

      for bnum lnum (length? ablk) 1 [
          if (found? find pick ablk bnum "Aminet recent uploads page") or
             (found? find pick ablk bnum "</pre>") [break]

          atmp: scanstr pick ablk bnum ">"
          if (pick atmp 1) = "<hr" [atmp: remove/part atmp 1]
          aname: substr pick atmp 2 1 ((pos "<" pick atmp 2) - 1)
          ainfo: substr pick atmp 3 2 ((pos "<" pick atmp 3) - 3)
          aread: substr pick atmp 4 1 ((pos "<" pick atmp 4) - 1)
          aread: replstr aread "&amp;" "&"
          insert tail foundtext form [aname ainfo aread]
          insert tail foundtext newline
      ]
      return head foundtext
   ]
]

]  ;; end of external USE statement
""

